#include "synchronize_lals_job.h"

#include "convert_lal_state_mapper.h"
#include "direct_entry_converter.h"
#include "synchronize_lals_mapper.h"

#include <crypta/lib/native/time/shifted_clock.h>
#include <crypta/lib/native/yt/utils/helpers.h>
#include <crypta/lib/native/yt/utils/timed_yt_path_generator.h>
#include <crypta/lib/proto/user_data/user_data_stats.pb.h>
#include <crypta/lookalike/lib/native/directory_finder.h>
#include <crypta/lookalike/proto/lal_state_key_value.pb.h>
#include <crypta/lookalike/proto/yt_node_names.pb.h>

#include <crypta/lookalike/services/lal_synchronizer/lib/synchronize_lals_reducer.h>
#include <crypta/lookalike/services/lal_synchronizer/proto/reducer_config.pb.h>

#include <mapreduce/yt/common/config.h>
#include <mapreduce/yt/interface/client.h>
#include <mapreduce/yt/util/temp_table.h>
#include <mapreduce/yt/util/ypath_join.h>

using namespace NCrypta::NLookalike;
using namespace NCrypta::NLookalike::NLalSynchronizer;

namespace {
    static const TString LAL_ID_FIELD = "lal_id";

    TString* GetClientTvmSecret(TReducerConfig& reducerConfig) {
        auto* credentials = reducerConfig.MutableWriter()->MutableCredentials();
        if (credentials->GetUseSecureTvm()) {
            return credentials->MutableTvm()->MutableClientTvmSecret();
        }
        return nullptr;
    }

    template<typename TConverter>
    void RunMap(NYT::ITransactionPtr tx, const TString& srcTablePath, const TString& dstTablePath, const TString& errorsTablePath) {
        using TMapper = TSynchronizeLalsMapper<TConverter>;
        auto spec = NYT::TMapOperationSpec().AddInput<typename TConverter::TEntry>(srcTablePath);

        typename TMapper::TOutputIndexes::TBuilder outputBuilder;
        outputBuilder.template AddOutput<TInputLalEntry>(spec, dstTablePath, TMapper::EOutputTables::Output);
        outputBuilder.template AddOutput<TError>(spec, errorsTablePath, TMapper::EOutputTables::Errors);

        tx->Map(spec, new TMapper(outputBuilder.GetIndexes()));
    }
}

int NLalSynchronizer::SynchronizeLalsJob(TConfig config, NLog::TLogPtr log) {
    Y_UNUSED(log);

    NYT::TConfig::Get()->Pool = config.GetYt().GetPool();

    const auto& errorsTablePath = TTimedYtPathGenerator(TShiftedClock::Now()).GetPath(config.GetErrorsDir());

    auto client = NYT::CreateClient(config.GetYt().GetProxy());
    auto tx = client->StartTransaction();

    {
        NYT::TTempTable convertedStateTable(tx, "converted_state_table", {}, NYT::TCreateOptions().Attributes(
            NYT::TNode()("schema", NYT::CreateTableSchema<TInputLalEntry>().ToNode())));

        const auto convertLalStateMapSpec = NYT::TMapOperationSpec()
            .AddInput<TLalStateKeyValue>(config.GetStateTable())
            .AddOutput<TInputLalEntry>(convertedStateTable.Name());

        tx->Map(convertLalStateMapSpec, new TConvertLalStateMapper());
        tx->Sort(convertedStateTable.Name(), convertedStateTable.Name(), {LAL_ID_FIELD});

        NYT::TTempTable intermediateTable(tx, "intermediate_lals");
        TString reduceSrcTablePath = intermediateTable.Name();

        const auto& scope = config.GetScope();
        if (scope == "direct") {
            RunMap<TDirectEntryConverter>(tx, config.GetInputTable(), reduceSrcTablePath, errorsTablePath);
            tx->Sort(reduceSrcTablePath, reduceSrcTablePath, {LAL_ID_FIELD});
        } else {
            reduceSrcTablePath = config.GetInputTable();
        }

        auto vault = NYT::TNode::CreateMap();
        auto* clientTvmSecret = GetClientTvmSecret(*config.MutableReducer());
        if (clientTvmSecret != nullptr) {
            vault["TVM_SECRET"] = *clientTvmSecret;
            clientTvmSecret->clear();
        }

        auto reduceSpec = NYT::TReduceOperationSpec()
            .AddOutput<TError>(NYT::TRichYPath(errorsTablePath).Append(true))
            .ReduceBy({LAL_ID_FIELD});

        TSynchronizeLalsReducer::TInputIndexes::TBuilder inputBuilder;
        inputBuilder.AddInput<TInputLalEntry>(reduceSpec, reduceSrcTablePath, TSynchronizeLalsReducer::EInputTables::Input);
        inputBuilder.AddInput<TInputLalEntry>(reduceSpec, convertedStateTable.Name(), TSynchronizeLalsReducer::EInputTables::State);

        tx->Reduce(reduceSpec, new TSynchronizeLalsReducer(config.GetReducer(), inputBuilder.GetIndexes()), NYT::TOperationOptions{}.SecureVault(vault));
    }

    tx->Commit();
    SetTtl(client, errorsTablePath, TDuration::Days(config.GetErrorsTtlDays()), ESetTtlMode::RemoveIfEmpty);

    return 0;
}

REGISTER_MAPPER(TSynchronizeLalsMapper<TDirectEntryConverter>);
