#include "synchronize_lals_reducer.h"

#include <crypta/lib/native/log/loggers/std_logger.h>
#include <crypta/lookalike/lib/native/pqlib_producer_initializer.h>
#include <crypta/lookalike/proto/error.pb.h>
#include <crypta/lookalike/services/lal_manager/commands/lal_cmd.pb.h>

#include <library/cpp/protobuf/json/proto2json.h>

#include <util/system/env.h>

using namespace NCrypta;
using namespace NCrypta::NLookalike;
using namespace NCrypta::NLookalike::NLalSynchronizer;

namespace {
    bool operator==(const TLalParent& lhs, const TLalParent& rhs) {
        return lhs.GetId() == rhs.GetId() && lhs.GetType() == rhs.GetType() && lhs.HasRule() == rhs.HasRule() && lhs.GetRule() == rhs.GetRule();
    }

    bool operator!=(const TLalParent& lhs, const TLalParent& rhs) {
        return !(lhs == rhs);
    }

    TLalParent GetParent(const TInputLalEntry& entry) {
        TLalParent parent;
        parent.SetType(entry.GetParentType());
        parent.SetId(entry.GetParentId());
        if (entry.HasParentRule()) {
            parent.SetRule(entry.GetParentRule());
        }
        return parent;
    }

}

TSynchronizeLalsReducer::TSynchronizeLalsReducer(TReducerConfig config, TInputIndexes inputIndexes)
    : Config(std::move(config))
    , InputIndexes(std::move(inputIndexes))
{}

void TSynchronizeLalsReducer::Start(TWriter* writer) {
    try {
        using namespace NCrypta::NPQ;

        Config.MutableWriter()->MutableCredentials()->MutableTvm()->SetClientTvmSecret(GetEnv("YT_SECURE_VAULT_TVM_SECRET"));

        auto log = NLog::NStdLogger::RegisterLog("pqlib", "stderr", "debug");

        Producer = GetPqlibProducer(Config.GetWriter());
    } catch (const std::exception& e) {
        WriteError(writer, e.what());
        throw;
    }
}

void TSynchronizeLalsReducer::Finish(TWriter* writer) {
    try {
        Producer->Stop(TDuration::Seconds(30));
    } catch (const yexception& e) {
        WriteError(writer, e.what());
        throw;
    }
}

void TSynchronizeLalsReducer::Do(TReader* reader, TWriter* writer) {
    ui64 lalId = 0;
    TMaybe<TLalParent> stateParent;
    TMaybe<TLalParent> inputParent;

    try {
        for (; reader->IsValid(); reader->Next()) {
            const auto& tableIndex = reader->GetTableIndex();
            const auto& row = reader->GetRow();
            lalId = row.GetLalId();

            if (tableIndex == InputIndexes[EInputTables::State]) {
                stateParent = GetParent(row);
            } else if (tableIndex == InputIndexes[EInputTables::Input]) {
                inputParent = GetParent(row);
            } else {
                ythrow yexception() << "Unknown table index: " << tableIndex;
            }
        }

        if (stateParent && !inputParent) {
            RemoveLal(lalId);
        } else if (inputParent && (!stateParent || (*stateParent != *inputParent))) {
            AddLal(lalId, std::move(*inputParent));
        }
    } catch(const yexception& e) {
        WriteError(writer, TStringBuilder() << "Lal id = " << lalId << ". Error = " << e.what());
    }
}

void TSynchronizeLalsReducer::AddLal(ui64 lalId, TLalParent&& parent) {
    TLalCmd cmd;
    auto& addLalCmd = *cmd.MutableAddLalCmd();
    addLalCmd.SetLalId(lalId);
    addLalCmd.MutableParent()->Swap(&parent);

    Y_ENSURE(Producer->TryEnqueue(NProtobufJson::Proto2Json(cmd)));
}

void TSynchronizeLalsReducer::RemoveLal(ui64 lalId) {
    TLalCmd cmd;
    cmd.MutableRemoveLalCmd()->SetLalId(lalId);

    Y_ENSURE(Producer->TryEnqueue(NProtobufJson::Proto2Json(cmd)));
}

void TSynchronizeLalsReducer::WriteError(TWriter* writer, const TString& message) {
    TError error;
    error.SetMessage(message);

    writer->AddRow(error);
}

REGISTER_REDUCER(TSynchronizeLalsReducer);
