#!/usr/bin/env python
# -*- coding: utf-8 -*-

import functools
import logging

import nirvana.job_context as nv

from crypta.lib.python.logging import logging_helpers
from crypta.lookalike.services.mobile_lookalike_training.lib import (
    affinitive_apps,
    apps_features_from_stores,
    apps_for_training,
    apps_publishers_vectors,
    apps_web_features,
    clusters,
    current_centroids,
    dssm_vectors,
    metrics_for_validation,
    train_sample,
    user_features_by_apps,
    users_for_training,
    users_web_features,
)


logger = logging.getLogger(__name__)


def main():
    logging_helpers.configure_stderr_logger(logging.getLogger(), level=logging.INFO)
    logger.info('Mobile LaL training')

    context = nv.context()
    parameters = context.parameters
    outputs = context.get_outputs()
    result_output = outputs.get('result')
    dict_output = outputs.get('dict_output')

    tasks_dict = {
        'calculate_affinitive_apps': functools.partial(affinitive_apps.calculate,
                                                       nv_params=parameters),

        'calculate_category_app2vec': functools.partial(apps_features_from_stores.calculate_category_app2vec,
                                                        nv_params=parameters),
        'get_apps_features_from_stores': functools.partial(apps_features_from_stores.get,
                                                           nv_params=parameters, result_output=result_output,
                                                           dict_output=dict_output),
        'get_data_from_stores': functools.partial(apps_features_from_stores.get_data_from_stores, nv_params=parameters),

        'select_apps_for_training': functools.partial(apps_for_training.select, nv_params=parameters),

        'get_apps_publishers_vectors': functools.partial(apps_publishers_vectors.get, nv_params=parameters),

        'get_apps_web_features': functools.partial(apps_web_features.get, nv_params=parameters,
                                                   output=dict_output),
        'prepare_apps_web_features': functools.partial(apps_web_features.prepare, nv_params=parameters),

        'get_apps_vectors_dssm': functools.partial(dssm_vectors.get_apps_vectors, nv_params=parameters),
        'get_users_features_dssm': functools.partial(dssm_vectors.get_users_features, nv_params=parameters),
        'get_users_vectors_dssm': functools.partial(dssm_vectors.get_users_vectors, nv_params=parameters),

        'calculate_distances': functools.partial(metrics_for_validation.calculate_distances, nv_params=parameters),
        'calculate_dssm_statistics': functools.partial(metrics_for_validation.calculate_dssm_statistics,
                                                       nv_params=parameters),
        'calculate_pr_auc': functools.partial(metrics_for_validation.calculate_pr_auc, nv_params=parameters),
        'calculate_random_statistics': functools.partial(metrics_for_validation.calculate_random_statistics,
                                                         nv_params=parameters),

        'make_train_sample': functools.partial(train_sample.make, nv_params=parameters),
        'merge_apps_features': functools.partial(train_sample.merge_apps_features, nv_params=parameters),

        'get_user_features_by_apps': functools.partial(user_features_by_apps.get, nv_params=parameters,
                                                       output=result_output),

        'get_users_for_training': functools.partial(users_for_training.get, nv_params=parameters),

        'get_users_web_features': functools.partial(users_web_features.get, nv_params=parameters,
                                                    output=result_output),

        'get_current_centroids': functools.partial(current_centroids.get, nv_params=parameters),

        'save_tables': functools.partial(metrics_for_validation.save_tables, nv_params=parameters),

        'update_clusters': functools.partial(clusters.update, nv_params=parameters),
    }

    job_name = parameters.get('job_name')
    logger.info('Job name: {}'.format(job_name))

    if job_name in tasks_dict:
        tasks_dict[job_name]()
    else:
        logger.warn('Unknown job_name "{}"'.format(job_name))
        exit(1)


if __name__ == '__main__':
    main()
