#!/usr/bin/env python
# -*- coding: utf-8 -*-

from functools import partial
import json
import logging

import numpy as np

from crypta.lib.python.nirvana.nirvana_helpers.nirvana_transaction import NirvanaTransaction
from crypta.lib.python.yt import yt_helpers
from crypta.lookalike.lib.python.utils import (
    fields,
    mobile_utils,
)
from crypta.lookalike.lib.python.utils.mobile_config import config as mobile_config
from crypta.lookalike.lib.python.utils.utils import get_feature_name

logger = logging.getLogger(__name__)


def get_data_from_stores(nv_params):
    yt_client = mobile_utils.get_yt_client(nv_params=nv_params)
    yql_client = mobile_utils.get_yql_client(nv_params=nv_params)

    with NirvanaTransaction(yt_client) as transaction:
        yql_client.execute(
            query=mobile_utils.get_data_from_stores_query.format(
                app_data=mobile_config.APP_DATA,
                merged_stores_raw=mobile_config.MERGED_STORES_RAW,
            ),
            transaction=str(transaction.transaction_id),
            title='YQL process data for apps from stores',
        )


def get_apps_features_mapper(row, feature_to_id):
    features = np.zeros(len(feature_to_id), dtype=np.float32)

    for feature_type in mobile_utils.categorical_features:
        if feature_type in row:
            if isinstance(row[feature_type], list):
                values = [str(item) for item in row[feature_type]]
            else:
                values = [row[feature_type]]

            for value in values:
                key = get_feature_name(feature_type, value).lower()
                if key in feature_to_id:
                    features[feature_to_id[key]] = 1.

    for feature_type in mobile_utils.float_features:
        if feature_type in row and row[feature_type] is not None:
            features[feature_to_id[feature_type]] = row[feature_type]
        else:
            features[feature_to_id[feature_type]] = mobile_utils.default_values_dict[feature_type]

    yield {
        fields.app_id: row[fields.app_id],
        fields.id_type: row[fields.id_type],
        fields.app_features_from_stores: map(lambda elem: float(elem), features),
    }


def get(nv_params, result_output, dict_output):
    yt_client = mobile_utils.get_yt_client(nv_params=nv_params)
    yql_client = mobile_utils.get_yql_client(nv_params=nv_params)

    with NirvanaTransaction(yt_client) as transaction, yt_client.TempTable() as apps_stats_table:
        yql_client.execute(
            query=mobile_utils.add_stats_data_for_apps_query.format(
                merged_stores_raw=mobile_config.MERGED_STORES_RAW,
                devid_by_app_table=mobile_config.DEVID_BY_APP_WITH_CRYPTA_ID,
                output_table=mobile_config.MERGED_STORES,
                MIN_DEVIDS_CNT=mobile_config.MIN_DEVIDS_COUNT,
                apps_stats=apps_stats_table,
            ),
            transaction=str(transaction.transaction_id),
            title='YQL add stats features for apps',
        )

        today_counts = list(yt_client.read_table(apps_stats_table))
        yt_helpers.write_stats_to_yt(
            yt_client=yt_client,
            table_path=mobile_config.DATALENS_MOBILE_LAL_COUNTERS_TABLE,
            data_to_write=today_counts,
            schema={
                'fielddate': 'string',
                'counter_name': 'string',
                'count': 'uint64',
            },
            date=mobile_utils.get_date_from_nv_parameters(nv_params=nv_params)
        )

        yql_client.execute(
            query=mobile_utils.features_table_query.format(
                input_table=mobile_config.MERGED_STORES,
                output_table=mobile_config.CATEGORICAL_FEATURES,
                min_count=mobile_config.MIN_APPS_COUNT,
            ),
            transaction=str(transaction.transaction_id),
            title='YQL get categorical features table',
        )
        logger.info('Categorical features are saved in the table.')

        feature_to_id = mobile_utils.get_features_dict(yt_client, mobile_config.CATEGORICAL_FEATURES)
        logger.info('Features number: {}'.format(len(feature_to_id)))

        yt_helpers.create_empty_table(
            yt_client=yt_client,
            path=mobile_config.APPS_FEATURES_FROM_STORES,
            schema={
                fields.app_id: 'string',
                fields.id_type: 'string',
                fields.app_features_from_stores: 'any',
            },
            force=True,
        )

        yt_client.run_map(
            partial(get_apps_features_mapper, feature_to_id=feature_to_id),
            mobile_config.MERGED_STORES,
            mobile_config.APPS_FEATURES_FROM_STORES,
            spec={
                'title': 'Map app features to a vector',
            },
        )
        logger.info('Apps features calculation is done.')

        with open(result_output, 'w') as output_file:
            json.dump({'ParamsSize': len(feature_to_id)}, output_file)

        logger.info('Successfully dumped features size')

        with open(dict_output, 'w') as output_file:
            json.dump(feature_to_id, output_file)

        logger.info('Successfully dumped features dict')


def calculate_category_app2vec(nv_params):
    yt_client = mobile_utils.get_yt_client(nv_params=nv_params)
    yql_client = mobile_utils.get_yql_client(nv_params=nv_params)

    with NirvanaTransaction(yt_client) as transaction, \
            yt_client.TempTable() as app_vector_category_table:
        yql_client.execute(
            query=mobile_utils.get_app_vector_category_query.format(
                merged_stores_table=mobile_config.MERGED_STORES,
                app2vec_table=mobile_config.APP2VEC_TABLE,
                output_table=app_vector_category_table,
            ),
            transaction=str(transaction.transaction_id),
            title='YQL get app-vector-category table',
        )

        yt_helpers.create_empty_table(
            yt_client=yt_client,
            path=mobile_config.CATEGORY2VEC_TABLE,
            schema={
                fields.category: 'string',
                fields.vector: 'any',
            },
            force=True,
        )

        yt_client.run_reduce(
            mobile_utils.reduce_apps_vectors_by_category,
            app_vector_category_table,
            mobile_config.CATEGORY2VEC_TABLE,
            reduce_by=[fields.category],
            spec={
                'title': 'Reduce app2vec for categories',
            },
        )
