#!/usr/bin/env python
# -*- coding: utf-8 -*-

import logging

from crypta.lib.python.nirvana.nirvana_helpers.nirvana_transaction import NirvanaTransaction
from crypta.lookalike.lib.python.utils import (
    fields,
    mobile_utils,
)
from crypta.lookalike.lib.python.utils.mobile_config import config as mobile_config

logger = logging.getLogger(__name__)

select_apps_for_validation_query = """
INSERT INTO `{output_table}`
WITH TRUNCATE

(
    SELECT *
    FROM `{dssm_app_vectors}`
    WHERE app_type == 'training'
    ORDER BY RANDOM(app_id)
    LIMIT {VALIDATION_APPS_CNT}
)
UNION ALL
(
    SELECT *
    FROM `{dssm_app_vectors}`
    WHERE app_type == 'validation'
    LIMIT {VALIDATION_APPS_CNT}
);
"""

users_features_query = """
$user_features = (
    SELECT
        mobile_features.id AS id,
        mobile_features.id_type AS id_type,
        apps,
        matching.cryptaId AS cryptaId,
        installed_apps,
        user_apps_vector_features,
        user_apps_features_from_stores,
        user_web_features
    FROM `{user_dssm_features_mobile}` AS mobile_features
    INNER JOIN `{matching}` AS matching
    ON mobile_features.id == matching.id AND mobile_features.id_type == matching.id_type
    LEFT JOIN `{user_dssm_web_features}` AS web_features
    ON matching.cryptaId == CAST(web_features.cryptaId AS String)
);

$default_vector = '{default_user_web_vector}';

INSERT INTO `{output_table}`
WITH TRUNCATE

SELECT
    id,
    id_type,
    cryptaId,
    apps,
    installed_apps,
    user_apps_vector_features,
    user_apps_features_from_stores,
    COALESCE(user_web_features, $default_vector) AS user_web_features
FROM $user_features;
"""


def get_apps_vectors(nv_params):
    yt_client = mobile_utils.get_yt_client(nv_params=nv_params)
    yql_client = mobile_utils.get_yql_client(nv_params=nv_params)

    with NirvanaTransaction(yt_client) as transaction:
        yql_client.execute(
            query=mobile_utils.apply_dssm_to_apps_query.format(
                id_column_name='app_id',
                additional_columns='{}, {},'.format(fields.app_type, fields.MD5Hash),
                released=mobile_config.RELEASED,
                input_table=mobile_config.APP_DSSM_FEATURES,
                output_table=mobile_config.APP_DSSM_VECTORS,
            ),
            transaction=str(transaction.transaction_id),
            title='YQL apply DSSM to all apps',
        )

        logger.info('Dssm vectors for all apps are calculated.')

        yql_client.execute(
            query=select_apps_for_validation_query.format(
                dssm_app_vectors=mobile_config.APP_DSSM_VECTORS,
                output_table=mobile_config.VALIDATION_APPS_DSSM_VECTORS,
                VALIDATION_APPS_CNT=mobile_config.VALIDATION_APPS_CNT,
            ),
            transaction=str(transaction.transaction_id),
            title='YQL select apps for validation',
        )
        logger.info('Validation apps are selected.')


def get_users_features(nv_params):
    yt_client = mobile_utils.get_yt_client(nv_params=nv_params)
    yql_client = mobile_utils.get_yql_client(nv_params=nv_params)

    with NirvanaTransaction(yt_client) as transaction:
        default_user_web_vector = list(yt_client.read_table(
            mobile_config.DEFAULT_USER_DSSM_FEATURES_WEB
        ))[0][fields.user_web_features]

        yql_client.execute(
            query=users_features_query.format(
                user_dssm_features_mobile=mobile_config.USERS_VALIDATION_FEATURES_MOBILE,
                user_dssm_web_features=mobile_config.USERS_DSSM_FEATURES_WEB,
                matching=mobile_config.APPS_BY_DEVID_AND_CRYPTA_ID,
                default_user_web_vector=default_user_web_vector,
                output_table=mobile_config.USERS_DSSM_FEATURES,
            ),
            transaction=str(transaction.transaction_id),
            title='YQL get user features for dssm',
        )
        logger.info('Users features for dssm are calculated.')


def get_users_vectors(nv_params):
    yt_client = mobile_utils.get_yt_client(nv_params=nv_params)
    yql_client = mobile_utils.get_yql_client(nv_params=nv_params)

    with NirvanaTransaction(yt_client) as transaction:
        yql_client.execute(
            query=mobile_utils.apply_dssm_to_users_query.format(
                input_table=mobile_config.USERS_DSSM_FEATURES,
                output_table=mobile_config.USERS_DSSM_VECTORS,
                additional_columns='{},'.format(fields.apps),
                released=mobile_config.RELEASED,
            ),
            transaction=str(transaction.transaction_id),
            title='YQL apply dssm to users',
        )
        logger.info('Dssm vectors for users are calculated.')
