#include <crypta/lib/native/log/loggers/std_logger.h>
#include <crypta/lib/native/proto_secrets/remove_secrets.h>
#include <crypta/lib/native/yaml/config/config.h>
#include <crypta/lookalike/lib/native/directory_finder.h>
#include <crypta/lookalike/proto/yt_node_names.pb.h>
#include <crypta/lookalike/services/model_supplier/proto/model_supplier_config.pb.h>

#include <mapreduce/yt/interface/client.h>
#include <mapreduce/yt/util/ypath_join.h>

#include <util/folder/dirut.h>
#include <util/generic/vector.h>

using namespace NCrypta;
using namespace NCrypta::NLookalike;
using namespace NCrypta::NLookalike::NModelSupplier;

namespace {
    void DropOldDirs(NYT::IClientBasePtr client, const TString& dir, size_t maxDirsCount, NLog::TLogPtr log) {
        auto dstSubDirs = client->List(dir);
        Sort(dstSubDirs, [](const auto& left, const auto& right) { return left.AsString() > right.AsString(); });

        for (size_t i = maxDirsCount; i < dstSubDirs.size(); ++i) {
            const auto& fullDirectoryPath = NYT::JoinYPaths(dir, dstSubDirs[i].AsString());
            log->info("Remove {}", fullDirectoryPath);
            client->Remove(fullDirectoryPath, NYT::TRemoveOptions().Recursive(true));
        }
    }
}

int main(int argc, const char** argv) {
    auto log = NLog::NStdLogger::RegisterLog("main", "stdout", "info");
    log->info("================ Start ================");

    try {
        const auto& config = ParseYamlConfig<TModelSupplierConfig>(argc, argv);

        log->info("Config:\n{}", NProtoSecrets::GetCopyWithoutSecrets(config).DebugString());

        auto client = NYT::CreateClient(config.GetYtProxy());
        auto tx = client->StartTransaction();

        const TVector<TString> modelFiles = {
            TYtNodeNames().GetDssmModelFile(),
            TYtNodeNames().GetSegmentsDictFile()
        };

        TVector<TString> nodesToCopy(modelFiles);
        nodesToCopy.push_back(TYtNodeNames().GetUserEmbeddingsTable());

        const auto& lastVersionSrcDir = NDirectoryFinder::FindLastWithChildren(tx, config.GetSrcDir(), modelFiles);

        log->info("Found last version dir: {}", lastVersionSrcDir);
        const auto& dstVersionDir = NYT::JoinYPaths(config.GetDstDir(), GetBaseName(lastVersionSrcDir));

        for (const auto& node : nodesToCopy) {
            const auto& srcNode = NYT::JoinYPaths(lastVersionSrcDir, node);
            const auto& dstNode = NYT::JoinYPaths(dstVersionDir, node);

            log->info("Copy from {} to {}", srcNode, dstNode);
            if (tx->Exists(srcNode)) {
                tx->Copy(srcNode, dstNode, NYT::TCopyOptions().Recursive(true).Force(true));
            }
        }

        DropOldDirs(tx, config.GetDstDir(), config.GetMaxDirsCount(), log);

        tx->Commit();

        log->info("================ Finish ================");

        return 0;
    } catch (const std::exception& e) {
        log->error("Failed with error: {}", e.what());
    }

    log->info("================ Error ================");

    return 1;
}
