#include "apply_segment_dssm_job.h"

#include <crypta/lib/native/retargeting_ids/retargeting_id.h>
#include <crypta/lib/native/time/shifted_clock.h>
#include <crypta/lib/native/yt/utils/helpers.h>
#include <crypta/lib/native/yt/utils/timed_yt_path_generator.h>
#include <crypta/lib/proto/user_data/user_data_stats.pb.h>
#include <crypta/lookalike/lib/native/directory_finder.h>
#include <crypta/lookalike/lib/native/goal_type_converter.h>
#include <crypta/lookalike/proto/lal_state_key_value.pb.h>
#include <crypta/lookalike/proto/mode.pb.h>
#include <crypta/lookalike/proto/parent_to_lal_mapping.pb.h>
#include <crypta/lookalike/proto/yt_node_names.pb.h>
#include <crypta/lookalike/services/segment_dssm_applier/lib/apply_segment_dssm_mapper.h>

#include <mapreduce/yt/common/config.h>
#include <mapreduce/yt/interface/client.h>
#include <mapreduce/yt/util/ypath_join.h>

#include <util/system/env.h>

using namespace NCrypta::NLookalike;
using namespace NCrypta::NLookalike::NSegmentDssmApplier;

namespace {
    void MakeParentToLalMapping(const NYT::IClientPtr& client, const TString& segmentsMetaPath, const TString& localPath) {
        auto reader = client->CreateTableReader<TSegmentMetaEntry>(segmentsMetaPath);
        TParentToLalMapping mapping;

        for (; reader->IsValid(); reader->Next()) {
            const auto& row = reader->GetRow();
            const auto& parent = row.GetMeta().GetParent();
            const auto& type = NGoalTypeConverter::ConvertLalParentTypeToRetargetingIdType(parent.GetType());
            if (type == NCrypta::NRetargetingIds::EType::Unknown) {
                continue;
            }
            const auto& retargetingId = NCrypta::NRetargetingIds::TRetargetingId(parent.GetId(), type).Serialize();
            (*mapping.MutableMap())[retargetingId] = row.GetSegmentId();
        }

        TFileOutput output(localPath);
        output << NCrypta::NProtoSerializer::ToJson(mapping);
    }

    TString* GetClientTvmSecret(TMapperConfig& mapperConfig) {
        auto* credentials = mapperConfig.MutableWriter()->MutableCredentials();
        if (credentials->GetUseSecureTvm()) {
            return credentials->MutableTvm()->MutableClientTvmSecret();
        }
        return nullptr;
    }
}

int NSegmentDssmApplier::ApplySegmentDssmJob(TConfig config, NLog::TLogPtr log) {
    NYT::TConfig::Get()->Pool = config.GetYt().GetPool();

    auto client = NYT::CreateClient(config.GetYt().GetProxy());
    auto tx = client->StartTransaction();

    const auto now = TShiftedClock::Now();
    const auto& errorsPath = TTimedYtPathGenerator(now).GetPath(config.GetErrorsDir());

    const auto& workingDir = NDirectoryFinder::FindLastWithChildren(tx, config.GetVersionsDir(), {
            TYtNodeNames().GetSegmentsDictFile(),
            TYtNodeNames().GetDssmModelFile(),
    });

    log->info("Found last version dir: {}", workingDir);

    auto spec = NYT::TMapOperationSpec();
    spec.AddInput<TLalStateKeyValue>(config.GetLalsTable());
    spec.MapperSpec(NYT::TUserJobSpec()
                            .AddFile(NYT::JoinYPaths(workingDir, TYtNodeNames().GetSegmentsDictFile()))
                            .AddFile(NYT::JoinYPaths(workingDir, TYtNodeNames().GetDssmModelFile())));

    NYT::TYPath embeddingsPath;
    NYT::TYPath segmentsMetaPath;

    const auto& mode = config.GetMapper().GetMode();

    log->info("Running in mode: {}", mode);

    if (mode == ModeValue::NEW) {
        embeddingsPath = TTimedYtPathGenerator(now).GetPath(NYT::JoinYPaths(workingDir, TYtNodeNames().GetFreshSegmentEmbeddingsDir()));
        segmentsMetaPath = TTimedYtPathGenerator(now).GetPath(NYT::JoinYPaths(workingDir, TYtNodeNames().GetFreshMetasDir()));
    } else {
        embeddingsPath = NYT::JoinYPaths(workingDir, TYtNodeNames().GetSegmentEmbeddingsTable());
        segmentsMetaPath = NYT::JoinYPaths(workingDir, TYtNodeNames().GetSegmentMetasTable());
    }

    const auto& outputTableIndexes = TApplySegmentDssmMapper::PrepareOutput(
            spec,
            embeddingsPath,
            segmentsMetaPath,
            errorsPath);

    log->info("Setting tvm secret");

    auto vault = NYT::TNode::CreateMap();
    auto* clientTvmSecret = GetClientTvmSecret(*config.MutableMapper());
    if (clientTvmSecret != nullptr) {
        vault["TVM_SECRET"] = *clientTvmSecret;
        clientTvmSecret->clear();
    }

    log->info("Starting map operation with TApplySegmentDssmMapper");

    tx->Map(spec, new TApplySegmentDssmMapper(config.GetMapper(), outputTableIndexes), NYT::TOperationOptions{}.SecureVault(vault));

    if (mode == ModeValue::NEW && EmptyTable(tx, embeddingsPath)) {
        log->info("Output is empty");
        log->info("Deleting segments embeddings: {}", embeddingsPath);
        tx->Remove(embeddingsPath);
        log->info("Deleting segments metas: {}", segmentsMetaPath);
        tx->Remove(segmentsMetaPath);
    }

    tx->Commit();

    SetTtl(client, errorsPath, TDuration::Days(config.GetErrorsTtlDays()), ESetTtlMode::RemoveIfEmpty);

    if (client->Exists(segmentsMetaPath)) {
        MakeParentToLalMapping(client, segmentsMetaPath, GetEnv("MAPPING_LOCAL_PATH"));
    }

    return 0;
}
