#include "make_segments_job.h"

#include <crypta/lib/native/time/shifted_clock.h>
#include <crypta/lib/native/yt/utils/helpers.h>
#include <crypta/lib/native/yt/utils/timed_yt_path_generator.h>
#include <crypta/lookalike/services/segmentator/lib/make_segments_mapper.h>
#include <crypta/lookalike/lib/native/directory_finder.h>
#include <crypta/lookalike/lib/native/get_date.h>
#include <crypta/lookalike/lib/native/get_oldest_timed_table.h>
#include <crypta/lookalike/proto/mode.pb.h>
#include <crypta/lookalike/proto/yt_node_names.pb.h>

#include <mapreduce/yt/common/config.h>
#include <mapreduce/yt/interface/client.h>
#include <mapreduce/yt/util/ypath_join.h>

using namespace NCrypta::NLookalike;

namespace {
    TString GetWorkingDir(const ModeValue& mode, const TString& versionsDir, NYT::IClientBasePtr client) {
        if (mode == ModeValue::NEW) {
            return NDirectoryFinder::FindLastWithChildren(client, versionsDir, {
                 TYtNodeNames().GetUserEmbeddingsTable(),
                 TYtNodeNames().GetFreshFilesDir(),
            });
        }

        return NDirectoryFinder::FindLastWithChildren(client, versionsDir, {
             TYtNodeNames().GetUserEmbeddingsTable(),
             TYtNodeNames().GetIndexFile(),
             TYtNodeNames().GetDataFile(),
             TYtNodeNames().GetLabelsFile()
        });
    }
}

int NSegmentator::MakeSegmentsJob(TMakeSegmentsJobConfig config, NLog::TLogPtr log) {
    NYT::TConfig::Get()->Pool = config.GetYtPool();

    auto client = NYT::CreateClient(config.GetYtProxy());
    auto tx = client->StartTransaction();

    const auto now = TShiftedClock::Now();
    const auto& errorsPath = TTimedYtPathGenerator(now).GetPath(config.GetErrorsDir());
    const auto& mode = config.GetMode();

    TString workingDir = GetWorkingDir(mode, config.GetVersionsDir(), tx);
    const auto date = GetDate(now);

    log->info("Found last version dir: {}", workingDir);

    NYT::TYPath indexFile;
    NYT::TYPath dataFile;
    NYT::TYPath labelsFile;
    NYT::TYPath userSegmentsTablePath;
    TString filesDir;

    if (mode == ModeValue::NEW) {
        TMaybe<TString> maybeFilesPath = GetOldestTimedTable(tx, NYT::JoinYPaths(workingDir, TYtNodeNames().GetFreshFilesDir()));

        auto userSegmentsDir = NYT::JoinYPaths(workingDir, TYtNodeNames().GetFreshUserSegmentsDir());
        if(!tx->Exists(userSegmentsDir)) {
            tx->Create(userSegmentsDir, NYT::NT_MAP, NYT::TCreateOptions().Recursive(true));
        }

        if (!maybeFilesPath) {
            tx->Commit();
            log->info("Fresh index file not found");
            log->info("================ Finish ================");
            return 0;
        }

        filesDir = NYT::JoinYPaths(workingDir, TYtNodeNames().GetFreshFilesDir(), *maybeFilesPath);
        userSegmentsTablePath = NYT::JoinYPaths(userSegmentsDir, *maybeFilesPath);
        indexFile = NYT::JoinYPaths(filesDir, TYtNodeNames().GetIndexFile());
        dataFile = NYT::JoinYPaths(filesDir, TYtNodeNames().GetDataFile());
        labelsFile = NYT::JoinYPaths(filesDir, TYtNodeNames().GetLabelsFile());
    } else {
        userSegmentsTablePath = NYT::JoinYPaths(workingDir, TYtNodeNames().GetUserSegmentsTable());
        indexFile = NYT::JoinYPaths(workingDir, TYtNodeNames().GetIndexFile());
        dataFile = NYT::JoinYPaths(workingDir, TYtNodeNames().GetDataFile());
        labelsFile = NYT::JoinYPaths(workingDir, TYtNodeNames().GetLabelsFile());

        if (IsProcessed(tx, indexFile)) {
            tx->Commit();
            log->info("Already processed for date: {}", date);
            log->info("================ Finish ================");
            return 0;
        }
    }

    auto spec = NYT::TMapOperationSpec();
    spec.AddInput<TUserEmbedding>(NYT::JoinYPaths(workingDir, TYtNodeNames().GetUserEmbeddingsTable()));
    spec.MapperSpec(NYT::TUserJobSpec()
                            .AddFile(indexFile)
                            .AddFile(dataFile)
                            .AddFile(labelsFile)
                            .AddEnvironment("Y_NO_AVX_IN_DOT_PRODUCT", config.GetDisableAvxInDotProduct() ? "1" : ""));

    const auto& outputTableIndexes = TMakeSegmentsMapper::PrepareOutput(spec, userSegmentsTablePath, errorsPath);

    log->info("Starting map operation with TMakeSegmentsMapper");
    tx->Map(spec, new TMakeSegmentsMapper(outputTableIndexes, config.GetMakeSegmentsMapper()));

    log->info("Starting merge operation for: {}", userSegmentsTablePath);
    tx->Merge(
        NYT::TMergeOperationSpec()
            .AddInput(userSegmentsTablePath)
            .Output(userSegmentsTablePath)
            .CombineChunks(true)
    );

    if (mode == ModeValue::NEW) {
        log->info("Removing processed files dir: {}", filesDir);
        tx->Remove(filesDir, NYT::TRemoveOptions().Recursive(true));
    } else {
        SetAttribute(tx, indexFile, LAST_PROCESSED_ATTR_NAME, date);
    }

    tx->Commit();

    SetTtl(client, errorsPath, TDuration::Days(config.GetErrorsTtlDays()), ESetTtlMode::RemoveIfEmpty);

    return 0;
}
