#!/usr/bin/env python
# -*- coding: utf-8 -*-

import functools
import logging

import nirvana.job_context as nv

from crypta.lib.python.logging import logging_helpers
from crypta.lookalike.services.top_common_lal_apps.lib import (
    apps_for_scoring,
    bb_format,
    distances,
    dssm_vectors,
    installers_by_ad,
    recommendations,
    segment_features,
    users,
    users_features,
)


logger = logging.getLogger(__name__)


def main():
    logging_helpers.configure_stderr_logger(logging.getLogger(), level=logging.INFO)
    logger.info('Mobile LaL application')

    context = nv.context()
    parameters = context.parameters
    inputs = context.get_inputs()
    dict_input = None
    if inputs.has('dict_input'):
        dict_input = inputs.get('dict_input')

    tasks_dict = {

        'get_daily_users': functools.partial(users.get_daily, nv_params=parameters),

        'get_top_and_promoted_apps': functools.partial(apps_for_scoring.get, nv_params=parameters),
        'get_installers_by_ad': functools.partial(installers_by_ad.get_segments, nv_params=parameters),

        'get_apps_features': functools.partial(segment_features.get_for_apps, nv_params=parameters,
                                               input_file=dict_input),
        'get_installers_by_ad_features': functools.partial(segment_features.get_for_installers_by_ad,
                                                           nv_params=parameters, input_file=dict_input),
        'get_user_features': functools.partial(users_features.get, nv_params=parameters, input_file=dict_input),

        'get_apps_vectors': functools.partial(dssm_vectors.get_for_apps, nv_params=parameters, inputs=inputs),
        'get_categories_vectors': functools.partial(dssm_vectors.get_for_categories, nv_params=parameters),
        'get_installers_by_ad_vectors': functools.partial(
            dssm_vectors.get_for_installers_by_ad,
            nv_params=parameters,
            inputs=inputs,
        ),
        'get_user_vectors': functools.partial(dssm_vectors.get_for_users, nv_params=parameters, inputs=inputs),

        'get_distances_for_categories': functools.partial(distances.get_for_categories, nv_params=parameters),
        'get_distances_for_installers_by_ad': functools.partial(distances.get_for_segments, nv_params=parameters),
        'get_distances_for_promoted_apps': functools.partial(distances.get_for_apps, promoted=True,
                                                             nv_params=parameters),
        'get_distances_for_top_apps': functools.partial(distances.get_for_apps, nv_params=parameters),

        'get_recommendations_for_apps': functools.partial(recommendations.get_for_apps_and_categories,
                                                          nv_params=parameters),
        'get_recommendations_for_segments': functools.partial(recommendations.get_for_segments, nv_params=parameters),

        'get_recommendations': functools.partial(bb_format.get, nv_params=parameters),
    }

    job_name = parameters.get('job_name')
    logger.info('Job name: {}'.format(job_name))

    if job_name in tasks_dict:
        tasks_dict[job_name]()
    else:
        logger.warn('Unknown job_name "{}"'.format(job_name))
        exit(1)


if __name__ == '__main__':
    main()
