import logging

from crypta.lib.python import templater
from crypta.lib.python.nirvana.nirvana_helpers.nirvana_transaction import NirvanaTransaction
from crypta.lookalike.lib.python.utils import mobile_utils
from crypta.lookalike.lib.python.utils.mobile_config import config as mobile_config

logger = logging.getLogger(__name__)

get_promoted_apps_query = """
$needed_data_from_stores = (
    SELECT
        BundleId,
        SourceID,
        MAX(CategoriesNames__raw) AS CategoriesNames__raw,
    FROM `{{app_data}}`
    WHERE RegionName == 'ru' AND SourceID IS NOT NULL
    GROUP BY BundleId, SourceID
);

$crypta_ids_counts = (
    SELECT
        id_type,
        app_id,
        COUNT(DISTINCT cryptaId) AS crypta_ids_cnt,
    FROM `{{devid_by_app_table}}`
    WHERE cryptaId IS NOT NULL
    GROUP BY app_id, id_type
);

$data_from_stores = (
    SELECT
        BundleId AS app_id,
        IF(SourceID == 1, 'gaid', 'idfa') AS id_type,
        ListHas(CategoriesNames__raw, 'GAMES') OR ListHas(CategoriesNames__raw, 'Games') AS game,
    FROM $needed_data_from_stores
);

$data_from_stores = (
    SELECT
        stores_data.*,
        counts.crypta_ids_cnt AS crypta_ids_cnt,
    FROM $data_from_stores AS stores_data
    INNER JOIN $crypta_ids_counts AS counts
    USING (app_id, id_type)
);

$promoted_apps = (
    SELECT DISTINCT
        app_id,
        IF(String::AsciiToLower(detaileddevicetype) LIKE '%ios%', 'idfa', 'gaid') AS id_type,
    FROM `{{campaign_stats_table}}`
    WHERE is_active
);

INSERT INTO `{{promoted_apps_table}}`
WITH TRUNCATE

SELECT
    data_from_stores.*,
FROM $promoted_apps AS promoted_apps
INNER JOIN $data_from_stores AS data_from_stores
USING (app_id, id_type);

"""


get_top_common_apps_query = """
INSERT INTO `{{top_common_apps_table}}`
WITH TRUNCATE

{% for id_type in id_types %}
{% for game_type in game_types %}
(
    SELECT
        app_id,
        id_type,
        game,
        crypta_ids_cnt,
        'top_{{id_type}}_{{game_type}}' AS source
    FROM $data_from_stores
    WHERE id_type == '{{id_type}}' AND game == {{game_type == 'game'}}
    ORDER BY crypta_ids_cnt DESC
    LIMIT {{counts[id_type]}}
)
{% if id_type != id_types[-1] or game_type != game_types[-1] %}
UNION ALL
{% endif %}
{% endfor %}
{% endfor %}
"""


def get(nv_params):
    yt_client = mobile_utils.get_yt_client(nv_params=nv_params)
    yql_client = mobile_utils.get_yql_client(nv_params=nv_params)

    if mobile_utils.check_date(yt_client, mobile_config.PROMOTED_APPS, nv_params) and \
            mobile_utils.check_date(yt_client, mobile_config.TOP_COMMON_APPS, nv_params):
        logger.info('Promoted and top apps table is ready for today')
        return

    with NirvanaTransaction(yt_client) as transaction:
        query = templater.render_template(
            get_promoted_apps_query + get_top_common_apps_query,
            vars=dict(
                app_data=mobile_config.APP_DATA,
                campaign_stats_table=mobile_config.CAMPAIGNS_STATISTICS,
                promoted_apps_table=mobile_config.PROMOTED_APPS,
                devid_by_app_table=mobile_config.DEVID_BY_APP_WITH_CRYPTA_ID_DAILY,
                id_types=['idfa', 'gaid'],
                game_types=['game', 'nogame'],
                counts={'idfa': mobile_config.DEVIDS_CNT_IDFA, 'gaid': mobile_config.DEVIDS_CNT_GAID},
                top_common_apps_table=mobile_config.TOP_COMMON_APPS,
            )
        )

        yql_client.execute(
            query,
            transaction=str(transaction.transaction_id),
            title='YQL get promoted apps',
        )

        mobile_utils.set_generate_date(yt_client, mobile_config.PROMOTED_APPS, nv_params)
        mobile_utils.set_generate_date(yt_client, mobile_config.TOP_COMMON_APPS, nv_params)
