import logging
import os
import time

from ads.bsyeti.libs.log_protos import crypta_profile_pb2
from yt.wrapper import with_context

from crypta.lib.python.nirvana.nirvana_helpers.nirvana_transaction import NirvanaTransaction
from crypta.lib.python.yt import yt_helpers
from crypta.lookalike.lib.python.utils import (
    fields,
    mobile_utils,
    utils,
)
from crypta.lookalike.lib.python.utils.mobile_config import config as mobile_config

logger = logging.getLogger(__name__)


@with_context
def convert_to_bb_format(key, records, context):
    timestamp = int(time.time())
    entry = {
        'timestamp': timestamp,
        'crypta_id': int(key[fields.cryptaId]),
    }

    bb_export_entry = crypta_profile_pb2.TCryptaLog(**entry)

    recommendations_record_cnt, segments_record_cnt = 0, 0
    for record in records:
        if context.table_index == mobile_config.NEW_RECOMMENDATIONS_TABLE_IDX:
            assert recommendations_record_cnt == 0, 'Too many apps recommendations rows for 1 crypta_id'
            recommendations_record_cnt += 1

            for keyword_id, field in (
                (mobile_config.TOP_COMMON_LAL_APPS_KEYWORD, fields.top_common_lal_apps),
                (mobile_config.PROMOTED_APPS_KEYWORD, fields.promoted),
            ):
                item = bb_export_entry.items.add(keyword_id=keyword_id, update_time=timestamp)
                for app_hash, score in record[field].items():
                    pair = item.pair_values.add()
                    pair.first = int(app_hash)
                    pair.second = int((score + 1) * 1000000)

            item = bb_export_entry.items.add(
                keyword_id=mobile_config.TOP_COMMON_LAL_CATEGORIES_KEYWORD,
                update_time=timestamp,
            )

            for category_id, score in record[fields.top_common_lal_categories].items():
                weighted_value = item.weighted_uint_values.add()
                weighted_value.first = int(category_id)
                weighted_value.weight = int((score + 1) * 1000000)
        else:
            assert segments_record_cnt == 0, 'Too many segment scores rows for 1 crypta_id'
            segments_record_cnt += 1

            item = bb_export_entry.items.add(
                keyword_id=mobile_config.CRYPTA_RMP_FEATURES_KEYWORD,
                update_time=timestamp,
            )
            for group_id, score in sorted(record[fields.installs_by_ads_scores].items()):
                item.uint_values.append(int((score + 1) * 1000000))

    yield {'value': bb_export_entry.SerializeToString()}


def get(nv_params):
    yt_client = utils.get_yt_client(nv_params=nv_params)

    with NirvanaTransaction(yt_client):
        today = mobile_utils.get_date_from_nv_parameters(nv_params)
        output_table = os.path.join(mobile_config.OUTPUT_DIRECTORY, today)

        if mobile_utils.check_date(yt_client, output_table, nv_params):
            logger.info('The tables for today are already ready.')
            return

        yt_helpers.create_empty_table(yt_client, output_table, schema={'value': 'string'}, force=True)

        yt_client.run_reduce(
            convert_to_bb_format,
            [
                mobile_config.NEW_RECOMMENDATIONS_TABLE,
                mobile_config.NEW_INSTALLS_BY_AD_SCORES,
            ],
            output_table,
            reduce_by=[fields.id_type, fields.cryptaId],
        )

        mobile_utils.set_generate_date(yt_client, output_table, nv_params)
        yt_helpers.set_yql_proto_field(output_table, 'value', crypta_profile_pb2.TCryptaLog, yt_client)

        logger.info('Output table for {} is calculated.'.format(today))

        while True:
            history_data = yt_client.list(mobile_config.OUTPUT_DIRECTORY, absolute=True)
            if len(history_data) > mobile_config.DAYS_TO_KEEP_RECOMMENDATIONS:
                yt_client.remove(min(history_data))
                logger.info('Table {} is removed.'.format(min(history_data)))
            else:
                break
