#!/usr/bin/env python
# -*- coding: utf-8 -*-

from functools import partial
import logging

from crypta.lib.python.nirvana.nirvana_helpers.nirvana_transaction import NirvanaTransaction
from crypta.lib.python.yt import yt_helpers
from crypta.lookalike.lib.python.utils import (
    fields,
    mobile_utils,
    yt_schemas,
)
from crypta.lookalike.lib.python.utils.mobile_config import config as mobile_config

logger = logging.getLogger(__name__)


def get_distances(yt_client, segments, output_table, output_table_schema, distances_mapper, nv_params):
    if mobile_utils.check_date(yt_client, output_table, nv_params):
        logger.info('{} table for today have already been prepared'.format(output_table))
        return

    yt_helpers.create_empty_table(
        yt_client=yt_client,
        path=output_table,
        schema=output_table_schema,
        additional_attributes={'optimize_for': 'scan'},
        force=True,
    )

    yt_client.run_map(
        partial(distances_mapper, segments=segments),
        mobile_config.USERS_VECTORS,
        output_table,
        spec={'data_size_per_job': 1024 * 1024 * 128},
    )

    yt_client.run_sort(output_table, sort_by=[fields.id_type, fields.cryptaId])
    mobile_utils.set_generate_date(yt_client, output_table, nv_params)


def get_for_segments(nv_params):
    yt_client = mobile_utils.get_yt_client(nv_params=nv_params)

    if mobile_utils.check_date(yt_client, mobile_config.INSTALLS_BY_AD_SCORES, nv_params,
                               gap_days=mobile_config.PERIOD_TO_CALCULATE_INSTALLERS_BY_AD):
        return

    with NirvanaTransaction(yt_client):
        segments = []
        for row in yt_client.read_table(mobile_config.INSTALLERS_BY_AD_VECTORS):
            segments.append({
                fields.group_id: row[fields.group_id],
                fields.vector: row[fields.segment_vector],
            })

        get_distances(
            yt_client,
            segments,
            mobile_config.INSTALLERS_BY_AD_LAL_DISTANCES,
            yt_schemas.lal_distances_segments_schema(),
            mobile_utils.segments_lal_scores_mapper,
            nv_params,
        )

    logger.info('Distances are calculated for installers by ad segments')


def get_for_apps(nv_params, promoted=False):
    yt_client = mobile_utils.get_yt_client(nv_params=nv_params)
    yql_client = mobile_utils.get_yql_client(nv_params=nv_params)

    lal_distances_table_path = mobile_config.PROMOTED_APPS_LAL_DISTANCES if promoted else mobile_config.TOP_APPS_LAL_DISTANCES
    info_table_path = mobile_config.PROMOTED_APPS if promoted else mobile_config.TOP_COMMON_APPS

    with NirvanaTransaction(yt_client) as transaction:
        segments = mobile_utils.get_apps(yt_client, yql_client, mobile_config.APPS_VECTORS, info_table_path, transaction)

        get_distances(
            yt_client,
            segments,
            lal_distances_table_path,
            yt_schemas.lal_distances_apps_schema(),
            mobile_utils.apps_lal_scores_mapper,
            nv_params,
        )

    logger.info('Distances are calculated for {} apps'.format('promoted' if promoted else 'top'))


def get_for_categories(nv_params):
    yt_client = mobile_utils.get_yt_client(nv_params=nv_params)

    with NirvanaTransaction(yt_client):
        categories = list(yt_client.read_table(mobile_config.CATEGORIES_VECTORS))
        get_distances(
            yt_client,
            categories,
            mobile_config.CATEGORIES_LAL_DISTANCES,
            yt_schemas.lal_distances_categories_schema(),
            mobile_utils.categories_lal_scores_mapper,
            nv_params,
        )

    logger.info('Distances are calculated for categories')
