#!/usr/bin/env python
# -*- coding: utf-8 -*-

import logging

from crypta.lib.python import templater
from crypta.lib.python.nirvana.nirvana_helpers.nirvana_transaction import NirvanaTransaction
from crypta.lookalike.lib.python.utils import (
    mobile_utils,
    utils,
)
from crypta.lookalike.lib.python.utils.mobile_config import config as mobile_config
from crypta.lookalike.proto import yt_node_names_pb2

logger = logging.getLogger(__name__)


aggregate_categories_vectors = """
INSERT INTO `{categories_vectors}`
WITH TRUNCATE

SELECT
    clusters.cluster_id AS cluster_id,
    clusters.id_type AS id_type,
    MULTI_AGGREGATE_BY(vectors.segment_vector, AggregationFactory("avg")) AS vector,
FROM `{clustering_table}` AS clusters
INNER JOIN `{apps_vectors_table}` AS vectors
ON vectors.GroupID == clusters.bundle_id || '__' || clusters.id_type
GROUP BY clusters.cluster_id, clusters.id_type;
"""


def get_for_segments(nv_params, input_table, output_table, inputs):
    yt_client = mobile_utils.get_yt_client(nv_params=nv_params)
    yql_client = mobile_utils.get_yql_client(nv_params=nv_params)
    yt_node_names = yt_node_names_pb2.TYtNodeNames()

    if mobile_utils.check_date(yt_client, output_table, nv_params):
        logger.info('Dssm vectors table {} for today is ready'.format(output_table))
        return

    with NirvanaTransaction(yt_client) as transaction:
        query = templater.render_template(
            utils.get_segments_embeddings_query_template,
            vars={
                'additional_fields': '',
                'model_path': utils.get_lal_model_source_link(
                    inputs=inputs,
                    file_name=yt_node_names.DssmModelSandboxName,
                ),
                'segments_dssm_features_table': input_table,
                'segments_dssm_vectors_table': output_table,
                'additional_dssm_fields': [],
            }
        )

        yql_client.execute(
            query=query,
            transaction=str(transaction.transaction_id),
            title='YQL LaL get dssm embeddings',
        )
        mobile_utils.set_generate_date(yt_client, output_table, nv_params)


def get_for_apps(nv_params, inputs):
    get_for_segments(nv_params, mobile_config.APPS_FEATURES, mobile_config.APPS_VECTORS, inputs)


def get_for_installers_by_ad(nv_params, inputs):
    yt_client = mobile_utils.get_yt_client(nv_params=nv_params)

    if mobile_utils.check_date(yt_client, mobile_config.INSTALLS_BY_AD_SCORES, nv_params,
                               gap_days=mobile_config.PERIOD_TO_CALCULATE_INSTALLERS_BY_AD):
        return
    get_for_segments(nv_params, mobile_config.INSTALLERS_BY_AD_FEATURES, mobile_config.INSTALLERS_BY_AD_VECTORS, inputs)


def get_for_users(nv_params, inputs):
    yt_client = mobile_utils.get_yt_client(nv_params=nv_params)
    yql_client = mobile_utils.get_yql_client(nv_params=nv_params)
    yt_node_names = yt_node_names_pb2.TYtNodeNames()

    if mobile_utils.check_date(yt_client, mobile_config.USERS_VECTORS, nv_params):
        logger.info('Dssm vectors table {} for today is ready'.format(mobile_config.USERS_VECTORS))
        return

    with NirvanaTransaction(yt_client) as transaction:
        query = templater.render_template(
            utils.get_users_embeddings_query_template,
            vars={
                'additional_fields': 'user_affinitive_apps AS apps,\nid_type,\ncryptaId,',
                'model_path': utils.get_lal_model_source_link(
                    inputs=inputs,
                    file_name=yt_node_names.DssmModelSandboxName,
                ),
                'users_dssm_features_table': mobile_config.USERS_FEATURES,
                'users_dssm_vectors_table': mobile_config.USERS_VECTORS,
                'additional_dssm_fields': [],
            }
        )

        yql_client.execute(
            query=query,
            transaction=str(transaction.transaction_id),
            title='YQL LaL get users dssm embeddings',
        )
        mobile_utils.set_generate_date(yt_client, mobile_config.USERS_VECTORS, nv_params)


def get_for_categories(nv_params):
    yt_client = mobile_utils.get_yt_client(nv_params=nv_params)
    yql_client = mobile_utils.get_yql_client(nv_params=nv_params)

    if mobile_utils.check_date(yt_client, mobile_config.CATEGORIES_VECTORS, nv_params):
        logger.info('Dssm vectors table {} for today is ready'.format(mobile_config.CATEGORIES_VECTORS))
        return

    with NirvanaTransaction(yt_client) as transaction:
        yql_client.execute(
            query=aggregate_categories_vectors.format(
                categories_vectors=mobile_config.CATEGORIES_VECTORS,
                clustering_table=mobile_config.CLUSTERING_TABLE,
                apps_vectors_table=mobile_config.APPS_VECTORS,
            ),
            transaction=str(transaction.transaction_id),
            title='YQL Aggregate apps categories vectors',
        )
        mobile_utils.set_generate_date(yt_client, mobile_config.CATEGORIES_VECTORS, nv_params)
