#!/usr/bin/env python
# -*- coding: utf-8 -*-

import logging


from crypta.lib.python.nirvana.nirvana_helpers.nirvana_transaction import NirvanaTransaction
from crypta.lookalike.lib.python.utils import (
    mobile_utils,
    utils,
)
from crypta.lookalike.lib.python.utils.mobile_config import config as mobile_config

logger = logging.getLogger(__name__)


installers_segments_by_crypta_id_query = """
$installs_joined = (
    SELECT
        UniqID,
        UniqIDSource,
        IDFA AS idfa,
        GoogleAdID AS gaid,
        OAID AS oaid,
        Goals_ID,
        Sign,
        GameQualifier AS is_game,
    FROM RANGE(`{installs_logs_dir}`, `{date_from}`, `{date_to}`) AS installs
    LEFT JOIN `{campaigns_table}` AS campaigns
    ON installs.OrderID == campaigns.orderid
    WHERE PlaceID == 542 AND EventType == 3 AND UniqID IS NOT NULL
);

$id_to_installs = (
    SELECT
        CASE
            WHEN gaid != '' THEN ('gaid', gaid)
            WHEN idfa != '' THEN ('idfa', idfa)
            WHEN oaid != '' THEN ('oaid', oaid)
            WHEN UniqIDSource == 'yandex-uid' THEN ('yandexuid', cast(UniqID AS String))
            ELSE NULL
        END AS id_tuple,
        is_game,
        SUM_IF(Sign, ListHas(Yson::ConvertToInt64List(Goals_ID), 4)) ?? 0 AS type4_installs_count,
        SUM_IF(Sign, ListHas(Yson::ConvertToInt64List(Goals_ID), 8)) ?? 0 AS type8_installs_count,
    FROM $installs_joined
    GROUP BY idfa, gaid, oaid, UniqID, UniqIDSource, is_game
);

$installers_by_ad = (
    SELECT
        Identifiers::Normalize(id_tuple.0, id_tuple.1) AS id,
        id_tuple.0 AS id_type,
        is_game,
        (type4_installs_count > 1) OR (type8_installs_count > 1) AS is_strong,
    FROM $id_to_installs
    WHERE Identifiers::Normalize(id_tuple.0, id_tuple.1) IS NOT NULL AND
        (type4_installs_count != 0 OR type8_installs_count != 0)
);

$installers_by_crypta_id = (
    SELECT
        ListNotNull(
            AsList(
                'all',
                IF(is_strong, 'strong_all', NULL),
                IF(is_game == 'game', 'games', NULL),
                IF(is_game == 'notGame', 'not_games', NULL),
                IF(is_strong and is_game == 'game', 'strong_games', NULL),
                IF(is_strong and is_game == 'notGame', 'strong_not_games', NULL),
            )
        ) AS segments,
        target_id AS crypta_id,
    FROM $installers_by_ad AS ids
    INNER JOIN CONCAT(
            `{gaid_matching_table}`,
            `{idfa_matching_table}`,
            `{oaid_matching_table}`,
            `{yandexuid_matching_table}`,
        ) AS matching
    USING (id_type, id)
);

INSERT INTO `{installers_segments_table}`
WITH TRUNCATE

SELECT
    crypta_id AS IdValue,
    'crypta_id' AS IdType,
    GroupID,
FROM $installers_by_crypta_id
FLATTEN LIST BY segments AS GroupID
"""


def get_segments(nv_params):
    yql_client = mobile_utils.get_yql_client(nv_params=nv_params)
    yt_client = mobile_utils.get_yt_client(nv_params=nv_params)

    if mobile_utils.check_date(yt_client, mobile_config.INSTALLERS_BY_AD_SEGMENTS, nv_params):
        logger.info('Installers by ad segments have already been calculated')
        return

    if mobile_utils.check_date(yt_client, mobile_config.INSTALLS_BY_AD_SCORES, nv_params,
                               gap_days=mobile_config.PERIOD_TO_CALCULATE_INSTALLERS_BY_AD):
        logger.info('Scores for installers by ad segments are not calculated today')
        return

    current_date = mobile_utils.get_date_from_nv_parameters(nv_params)

    with NirvanaTransaction(yt_client) as transaction:
        yql_client.execute(
            query=installers_segments_by_crypta_id_query.format(
                installs_logs_dir=mobile_config.MOBILE_POSTCLICK_LOG,
                date_to=current_date,
                date_from=utils.get_date_from_past(current_date, months=mobile_config.MONTHS_FOR_INSTALLERS_BY_AD),
                campaigns_table=mobile_config.CAMPAIGNS_STATISTICS,
                gaid_matching_table=mobile_config.GAID_TO_CRYPTA_ID_MATCHING,
                idfa_matching_table=mobile_config.IDFA_TO_CRYPTA_ID_MATCHING,
                oaid_matching_table=mobile_config.OAID_TO_CRYPTA_ID_MATCHING,
                yandexuid_matching_table=mobile_config.YUID_TO_CRYPTA_ID_MATCHING,
                installers_segments_table=mobile_config.INSTALLERS_BY_AD_SEGMENTS,
            ),
            transaction=str(transaction.transaction_id),
            title='YQL Make segments of installers by ads',
        )

        mobile_utils.set_generate_date(yt_client, mobile_config.INSTALLERS_BY_AD_SEGMENTS, nv_params)
