#!/usr/bin/env python
# -*- coding: utf-8 -*-

import logging

from crypta.lib.python import templater
from crypta.lib.python.nirvana.nirvana_helpers.nirvana_transaction import NirvanaTransaction
from crypta.lib.python.yt import yt_helpers
from crypta.lookalike.lib.python.utils import (
    fields,
    mobile_utils,
    utils,
    yt_schemas,
)
from crypta.lookalike.lib.python.utils.mobile_config import config as mobile_config

logger = logging.getLogger(__name__)


crypta_ids_counts_query = """
$data = (
    SELECT
        cryptaId,
        {id_field}
    FROM `{input_table}`
    FLATTEN LIST BY (DictKeys(Yson::ConvertToDoubleDict({field})) AS {id_field})
);

$counts = (
    SELECT
        {id_field},
        COUNT(*) AS crypta_ids_cnt,
    FROM $data
    GROUP BY {id_field}
);

INSERT INTO `{output_table}`
WITH TRUNCATE

SELECT
    PERCENTILE(crypta_ids_cnt, 0.1) AS pct_10_devids_cnt,
    PERCENTILE(crypta_ids_cnt, 0.25) AS pct_25_devids_cnt,
    PERCENTILE(crypta_ids_cnt, 0.5) AS median_devids_cnt,
    PERCENTILE(crypta_ids_cnt, 0.75) AS pct_75_devids_cnt,
FROM $counts;
"""


get_lower_bounds_query_template = """
INSERT INTO `{{lower_bounds_table}}`
WITH TRUNCATE

SELECT
{% for metric in metrics %}
    {{ coefficient }} * AVG({{ metric }}) AS {{ metric }},
{% endfor %}
FROM (
    SELECT *
    FROM `{{ counts_table }}`
    ORDER BY `timestamp` DESC
    LIMIT {{ days_to_count }}
);
"""


def check_metrics(yt_client, yql_client, transaction, current_metrics):
    with yt_client.TempTable() as lower_bounds_table:
        get_lower_bounds_query = templater.render_template(
            template_text=get_lower_bounds_query_template,
            vars={
                'lower_bounds_table': lower_bounds_table,
                'metrics': mobile_config.PROMOTED_APPS_RECOMMENDATIONS_METRICS,
                'coefficient': 0.66,
                'days_to_count': min(mobile_config.DAYS_TO_COUNT,
                                     yt_client.row_count(mobile_config.PROMOTED_APPS_RECOMMENDATION_COUNTS)),
                'counts_table': mobile_config.PROMOTED_APPS_RECOMMENDATION_COUNTS,
            },
        )

        yql_client.execute(
            query=get_lower_bounds_query,
            transaction=str(transaction.transaction_id),
            title='YQL get lower bounds metrics.',
        )

        lower_bounds = list(yt_client.read_table(lower_bounds_table))[0]
        for counter_name, count in lower_bounds.items():
            logger.info('Lower bound for {} is {}'.format(counter_name, count))

        accepted_metrics_count = 0
        for counter_name in mobile_config.PROMOTED_APPS_RECOMMENDATIONS_METRICS:
            if lower_bounds[counter_name] <= current_metrics[counter_name]:
                accepted_metrics_count += 1
            else:
                logger.info(
                    'Metric {} is below the accepted value: '
                    'recommended_count={} < lower_bound={}'.format(
                        counter_name, lower_bounds[counter_name], current_metrics[counter_name],
                    )
                )
        # metrics are considered unacceptable if their values have decreased by a third of of their average values
        assert accepted_metrics_count > 0, 'All metrics are too low to save recommendations.'


def get_and_check_counts(yt_client, transaction, nv_params):
    yql_client = mobile_utils.get_yql_client(nv_params=nv_params)

    with yt_client.TempTable() as counts_table:
        for field, id_field in ((fields.top_common_lal_apps, fields.MD5Hash),
                                (fields.promoted, fields.MD5Hash),
                                (fields.top_common_lal_categories, fields.cluster_id)):
            yql_client.execute(
                query=crypta_ids_counts_query.format(
                    input_table=mobile_config.NEW_RECOMMENDATIONS_TABLE,
                    field=field,
                    id_field=id_field,
                    output_table=counts_table,
                ),
                transaction=str(transaction.transaction_id),
                title='YQL calculation of crypta_id number for apps in recommendations.',
            )

            counts = list(yt_client.read_table(counts_table))[0]
            metrics_to_send = []
            for counter_name, count in counts.items():
                metrics_to_send.append({
                    'counter_name': utils.get_feature_name(counter_name, field),
                    'count': float(count),
                })

            yt_helpers.write_stats_to_yt(
                yt_client=yt_client,
                table_path=mobile_config.DATALENS_MOBILE_LAL_RECOMMENDATIONS_TABLE,
                data_to_write=metrics_to_send,
                schema={
                    'fielddate': 'string',
                    'counter_name': 'string',
                    'count': 'double',
                },
                date=mobile_utils.get_date_from_nv_parameters(nv_params=nv_params),
            )

            if field == fields.promoted:
                counts.update({'timestamp': str(nv_params['timestamp'])})
                yt_client.write_table(yt_client.TablePath(
                    mobile_config.PROMOTED_APPS_RECOMMENDATION_COUNTS, append=True,
                ), [counts])

                check_metrics(yt_client, yql_client, transaction, current_metrics=counts)


def get_for_apps_and_categories(nv_params):
    yt_client = mobile_utils.get_yt_client(nv_params=nv_params)

    with NirvanaTransaction(yt_client) as transaction:
        if mobile_utils.check_date(yt_client, mobile_config.NEW_RECOMMENDATIONS_TABLE, nv_params):
            logger.info('Recommendations for today have already been prepared.')
            return

        yt_helpers.create_empty_table(
            yt_client=yt_client,
            path=mobile_config.NEW_RECOMMENDATIONS_TABLE,
            schema=yt_schemas.get_recommendations_table_schema(),
            additional_attributes={'optimize_for': 'scan'},
            force=True,
        )
        input_tables = [
            mobile_config.PROMOTED_APPS_LAL_DISTANCES,
            mobile_config.TOP_APPS_LAL_DISTANCES,
            mobile_config.CATEGORIES_LAL_DISTANCES,
        ]

        yt_client.run_reduce(
            mobile_utils.apps_scores_reducer,
            input_tables,
            mobile_config.NEW_RECOMMENDATIONS_TABLE,
            reduce_by=[fields.id_type, fields.cryptaId],
            spec={'combine_chunks': True},
        )

        yt_client.run_sort(mobile_config.NEW_RECOMMENDATIONS_TABLE, sort_by=[fields.id_type, fields.cryptaId])

        logger.info('New recommendations are calculated for all apps and clusters')
        mobile_utils.set_generate_date(yt_client, mobile_config.NEW_RECOMMENDATIONS_TABLE, nv_params)

        if mobile_config.is_production:
            get_and_check_counts(yt_client, transaction, nv_params)


def get_for_segments(nv_params):
    yt_client = mobile_utils.get_yt_client(nv_params=nv_params)

    if mobile_utils.check_date(yt_client, mobile_config.NEW_INSTALLS_BY_AD_SCORES, nv_params):
        logger.info('New installs scores for today have already been prepared.')
        return

    with NirvanaTransaction(yt_client):
        if mobile_utils.check_date(
            yt_client,
            mobile_config.NEW_INSTALLS_BY_AD_SCORES,
            nv_params,
            gap_days=mobile_config.PERIOD_TO_CALCULATE_INSTALLERS_BY_AD,
        ):
            logger.info('New installs scores are not calculated today')
            return

        yt_helpers.create_empty_table(
            yt_client=yt_client,
            path=mobile_config.NEW_INSTALLS_BY_AD_SCORES,
            schema=yt_schemas.get_installers_by_ad_scores_table_schema(),
            additional_attributes={'optimize_for': 'scan'},
            force=True,
        )

        yt_client.run_reduce(
            mobile_utils.installers_segments_scores_reducer,
            mobile_config.INSTALLERS_BY_AD_LAL_DISTANCES,
            mobile_config.NEW_INSTALLS_BY_AD_SCORES,
            reduce_by=[fields.id_type, fields.cryptaId],
            spec={'combine_chunks': True},
        )

        yt_client.run_sort(mobile_config.NEW_INSTALLS_BY_AD_SCORES, sort_by=[fields.id_type, fields.cryptaId])

        logger.info('Scores are calculated for installers by ads segments')
        mobile_utils.set_generate_date(yt_client, mobile_config.NEW_INSTALLS_BY_AD_SCORES, nv_params)
