#!/usr/bin/env python
# -*- coding: utf-8 -*-

import datetime
import logging

from crypta.lib.python import templater
from crypta.lib.python.nirvana.nirvana_helpers.nirvana_transaction import NirvanaTransaction
from crypta.lookalike.lib.python.utils import utils
from crypta.lookalike.lib.python.utils.config import (
    config,
    environment,
)


logger = logging.getLogger(__name__)

get_last_updated_dates_template = """
{% for id_name in id_names %}
${{id_name}}_updated_date = (
    SELECT
        MAX(date_last_used)
    FROM `{{'{' + id_name + '_table}'}}`
);
{% endfor %}

INSERT INTO `{update_dates_table}`
WITH TRUNCATE
SELECT
{% for id_name in id_names %}
    ${{id_name}}_updated_date AS {{id_name}}_updated_date,
{% endfor %};
"""

update_ads_table_template = """
$date_to = CAST(CurrentUtcDate() AS String);
{% for id_name, log_id_column_name, log_date_column_name, flatten_needed in ads_ids_types %}
$date_from = (
    SELECT
        {{id_name}}_updated_date
    FROM `{update_dates_table}`
);

$new_{{id_name}} = (
    SELECT
        CAST({{log_id_column_name}} AS Uint64) AS goal_id,
        {{log_date_column_name}} AS date_last_used
    FROM RANGE(`{{'{' + id_name + '_logs_dir}'}}`, $date_from)
    {% if flatten_needed %}
    FLATTEN LIST BY ({{log_id_column_name}})
    {% endif %}
UNION ALL
    SELECT *
    FROM `{{'{' + id_name + '_table}'}}`
);

INSERT INTO `{{'{' + id_name + '_table}'}}`
WITH TRUNCATE
SELECT
    goal_id,
    MAX(date_last_used) AS date_last_used
FROM $new_{{id_name}}
WHERE date_last_used > '{{'{last_date_for_' + id_name + '}'}}'
GROUP BY goal_id
ORDER BY date_last_used DESC;
{% endfor %}

"""

split_goals_and_segments = """
$autobudget_goals_date_to = CAST('{current_date}' AS Date);

$get_goal_type = ($goal_id) -> {{
    RETURN
        CASE
            WHEN $goal_id >= {goals_ids_start} AND $goal_id < {goals_ids_end} THEN 'goal'
            WHEN $goal_id >= {audience_ids_start} AND $goal_id < {audience_ids_end} THEN 'audience'
            ELSE 'other'
        END
}};

$get_date = ($time) -> {{
    RETURN DateTime::MakeDate(DateTime::FromSeconds(CAST($time AS Uint32)))
}};

$autobudget_goals = (
    SELECT
        GoalID AS goal_id,
        MAX(CAST($get_date(LastUpdateTime) AS String)) AS date_last_used
    FROM `{autobudget_goals_log_table}`
    WHERE GoalID != -1 AND
        $get_date(LastUpdateTime) > CAST('{autobudget_goals_date_from}' AS Date) AND
        $get_date(LastUpdateTime) < $autobudget_goals_date_to
    GROUP BY GoalID
);

$all_goals = (
    SELECT
        goal_id,
        date_last_used,
        'meaningful_goal' AS ad_type
    FROM `{meaningful_goals_ids_table}`
    WHERE goal_id is not null
UNION ALL
    SELECT
        goal_id,
        date_last_used,
        'retargeting_goal' AS ad_type
    FROM `{retargeting_ids_table}`
    WHERE goal_id IS NOT null AND $get_goal_type(goal_id) == 'goal'
UNION ALL
    SELECT
        CAST(goal_id AS uint64) AS goal_id,
        date_last_used,
        'multiplier_goal' AS ad_type
    FROM `{multipliers_ids_table}`
    WHERE goal_id IS NOT null AND $get_goal_type(goal_id) == 'goal'
UNION ALL
    SELECT
        CAST(goal_id AS uint64) AS goal_id,
        date_last_used,
        'autobudget_goal' AS ad_type
    FROM $autobudget_goals
    WHERE goal_id IS NOT null
);

INSERT INTO `{training_goals_table}`
WITH TRUNCATE
SELECT
    goal_id,
    MAX(date_last_used) AS date_last_used,
    AGGREGATE_LIST_DISTINCT(ad_type) AS ad_types
FROM $all_goals
GROUP BY goal_id
ORDER BY date_last_used DESC;

$all_segments = (
    SELECT
        goal_id,
        date_last_used,
        'retargeting_segment' AS ad_type
    FROM `{retargeting_ids_table}`
    WHERE goal_id is not null AND $get_goal_type(goal_id) == 'audience'
UNION ALL
    SELECT
        CAST(goal_id as uint64) AS goal_id,
        date_last_used,
        'multiplier_segment' AS ad_type
    FROM `{multipliers_ids_table}`
    WHERE goal_id is not null AND $get_goal_type(goal_id) == 'audience'
);

INSERT INTO `{training_segments_table}`
WITH TRUNCATE
SELECT
    goal_id - {audience_ids_start} AS segment_id,
    MAX(date_last_used) AS date_last_used,
    AGGREGATE_LIST_DISTINCT(ad_type) AS ad_types
FROM $all_segments
GROUP BY goal_id
ORDER BY date_last_used DESC;
"""


def update(nv_params):
    yt_client = utils.get_yt_client(nv_params=nv_params)
    yql_client = utils.get_yql_client(nv_params=nv_params)

    current_date = str(datetime.date.today()) if environment.environment == 'production' else '2021-08-21'

    with NirvanaTransaction(yt_client) as transaction:
        get_last_updated_dates = templater.render_template(
            template_text=get_last_updated_dates_template,
            vars={
                'id_names': ['meaningful_goals', 'retargeting_ids', 'multipliers_ids'],
            },
        )
        yql_client.execute(
            query=get_last_updated_dates.format(
                meaningful_goals_table=config.MEANINGFUL_GOALS_IDS_TABLE,
                retargeting_ids_table=config.RETARGETING_IDS_TABLE,
                multipliers_ids_table=config.MULTIPLIERS_IDS_TABLE,
                update_dates_table=config.ADS_UPDATE_DATES_TABLE,
            ),
            transaction=str(transaction.transaction_id),
            title='YQL Get last updated dates for ads goals and segments tables',
        )

        ads_ids_info = [
            ('meaningful_goals', 'campaign_meaningful_goals', 'cube_date', True),
            ('retargeting_ids', 'retargeting_id', '`date`', False),
            ('multipliers_ids', 'retargeting_id', '`date`', False),
        ]
        update_ads_ids_tables = templater.render_template(
            template_text=update_ads_table_template,
            vars={
                'ads_ids_types': ads_ids_info,
            },
        )
        yql_client.execute(
            query=update_ads_ids_tables.format(
                update_dates_table=config.ADS_UPDATE_DATES_TABLE,
                meaningful_goals_logs_dir=config.MEANINGFUL_GOALS_DIR,
                meaningful_goals_table=config.MEANINGFUL_GOALS_IDS_TABLE,
                retargeting_ids_logs_dir=config.RETARGETING_DIR,
                retargeting_ids_table=config.RETARGETING_IDS_TABLE,
                multipliers_ids_logs_dir=config.MULTIPLIERS_DIR,
                multipliers_ids_table=config.MULTIPLIERS_IDS_TABLE,
                last_date_for_meaningful_goals=utils.get_date_from_past(current_date=current_date, months=1),
                last_date_for_retargeting_ids=utils.get_date_from_past(current_date=current_date, months=6),
                last_date_for_multipliers_ids=utils.get_date_from_past(current_date=current_date, months=6),
            ),
            transaction=str(transaction.transaction_id),
            title='YQL Update meaningful goals, retargeting and multipliers tables',
        )

        yql_client.execute(
            query=split_goals_and_segments.format(
                current_date=current_date,
                autobudget_goals_log_table=config.AUTOBUDGET_GOALS_LOG_TABLE,
                autobudget_goals_date_from=utils.get_date_from_past(current_date=current_date, months=4),
                meaningful_goals_ids_table=config.MEANINGFUL_GOALS_IDS_TABLE,
                retargeting_ids_table=config.RETARGETING_IDS_TABLE,
                multipliers_ids_table=config.MULTIPLIERS_IDS_TABLE,
                audience_ids_start=config.AUDIENCE_SEGMENTS_IDS_START,
                audience_ids_end=config.AUDIENCE_SEGMENTS_IDS_END,
                goals_ids_start=config.GOALS_IDS_START,
                goals_ids_end=config.GOALS_IDS_END,
                training_goals_table=config.TRAINING_GOALS_TABLE,
                training_segments_table=config.TRAINING_SEGMENTS_TABLE,
            ),
            transaction=str(transaction.transaction_id),
            title='YQL Split ids to segments and goals tables',
        )
