#include "apply_user_dssm_job.h"

#include <crypta/lib/native/time/shifted_clock.h>
#include <crypta/lib/native/yt/utils/helpers.h>
#include <crypta/lib/native/yt/utils/timed_yt_path_generator.h>
#include <crypta/lib/proto/user_data/user_data.pb.h>
#include <crypta/lookalike/lib/native/directory_finder.h>
#include <crypta/lookalike/proto/yt_node_names.pb.h>
#include <crypta/lookalike/services/user_dssm_applier/cpp/register.h>
#include <crypta/lookalike/services/user_dssm_applier/mapper/apply_user_dssm_mapper.h>

#include <mapreduce/yt/common/config.h>
#include <mapreduce/yt/interface/client.h>
#include <mapreduce/yt/util/ypath_join.h>

using namespace NCrypta;
using namespace NCrypta::NLookalike;

namespace {
    constexpr const char * const LAST_UPDATE_DATE = "@_last_update_date";
    constexpr const char * const ARCADIA_REVISION = "@arcadia_revision";

    NYT::TNode GetAttribute(NYT::IClientBasePtr client, const TString& path, const char * const attribute) {
        const auto& attributePath = NYT::JoinYPaths(path, attribute);
        if (!client->Exists(attributePath)) {
            return NYT::TNode();
        }
        return client->Get(attributePath);
    }

    NYT::TNode GetLastUpdateDate(NYT::IClientBasePtr client, const TString& path) {
        return GetAttribute(client, path, LAST_UPDATE_DATE);
    }

    NYT::TNode GetArcadiaRevision(NYT::IClientBasePtr client, const TString& path) {
        return GetAttribute(client, path, ARCADIA_REVISION);
    }

    void SetAttribute(NYT::IClientBasePtr client, const TString& path, const char * const attribute, const NYT::TNode& value) {
        client->Set(NYT::JoinYPaths(path, attribute), value);
    }

    void SetLastUpdateDate(NYT::IClientBasePtr client, const TString& path, const NYT::TNode& lastUpdateDate) {
        SetAttribute(client, path, LAST_UPDATE_DATE, lastUpdateDate);
    }

    void SetArcadiaRevision(NYT::IClientBasePtr client, const TString& path, const NYT::TNode& arcadiaRevision) {
        SetAttribute(client, path, ARCADIA_REVISION, arcadiaRevision);
    }
}

int NUserDssmApplier::ApplyUserDssmJob(TConfig config, NLog::TLogPtr log) {
    NYT::TConfig::Get()->Pool = config.GetYt().GetPool();

    auto client = NYT::CreateClient(config.GetYt().GetProxy());
    auto tx = client->StartTransaction();

    const auto now = TShiftedClock::Now();
    const auto& errorsPath = TTimedYtPathGenerator(now).GetPath(config.GetErrorsDir());

    const auto& versionsDir = config.GetVersionsDir();
    const auto& workingDir = NDirectoryFinder::FindLastWithChildren(tx, versionsDir, {
            TYtNodeNames().GetSegmentsDictFile(),
            TYtNodeNames().GetDssmModelFile(),
    });

    log->info("Found last version dir: {}", workingDir);

    const auto& userEmbeddingsPath = NYT::JoinYPaths(workingDir, TYtNodeNames().GetUserEmbeddingsTable());
    const auto& lastUpdateDate = GetLastUpdateDate(tx, config.GetUserDataTable());
    Y_ENSURE(!lastUpdateDate.IsUndefined(), "UserData has no last update date");

    if (tx->Exists(userEmbeddingsPath) && GetLastUpdateDate(tx, userEmbeddingsPath) == lastUpdateDate) {
        log->info("Embeddings are up to date, exiting.");
        return 0;
    }

    const auto& dssmModelFile = NYT::JoinYPaths(workingDir, TYtNodeNames().GetDssmModelFile());
    auto spec = NYT::TMapOperationSpec();
    spec.AddInput<NLab::TUserData>(config.GetUserDataTable());
    spec.MapperSpec(NYT::TUserJobSpec()
                            .AddFile(NYT::JoinYPaths(workingDir, TYtNodeNames().GetSegmentsDictFile()))
                            .AddFile(dssmModelFile));

    const auto& outputTableIndexes = TApplyUserDssmMapper::PrepareOutput(spec, userEmbeddingsPath, errorsPath);

    tx->Map(spec, new TApplyUserDssmMapper(outputTableIndexes));

    const auto& arcadiaRevision = GetArcadiaRevision(tx, dssmModelFile);
    Y_ENSURE(!arcadiaRevision.IsUndefined(), "Dssm model has no arcadia revision");

    SetArcadiaRevision(tx, userEmbeddingsPath, arcadiaRevision);
    SetArcadiaRevision(tx, versionsDir, arcadiaRevision);
    SetLastUpdateDate(tx, userEmbeddingsPath, lastUpdateDate);
    SetLastUpdateDate(tx, versionsDir, lastUpdateDate);

    tx->Commit();

    SetTtl(client, errorsPath, TDuration::Days(config.GetErrorsTtlDays()), ESetTtlMode::RemoveIfEmpty);

    return 0;
}
