#include "export_user_segments_job.h"

#include "export_user_segments_mapper.h"
#include "export_user_segments_reducer.h"

#include <crypta/lib/native/time/shifted_clock.h>
#include <crypta/lib/native/yt/utils/helpers.h>
#include <crypta/lib/native/yt/utils/timed_yt_path_generator.h>
#include <crypta/lookalike/lib/native/directory_finder.h>
#include <crypta/lookalike/lib/native/get_oldest_timed_table.h>
#include <crypta/lookalike/proto/audience_user_segment.pb.h>
#include <crypta/lookalike/proto/mode.pb.h>
#include <crypta/lookalike/proto/segment_meta_entry.pb.h>
#include <crypta/lookalike/proto/yt_node_names.pb.h>

#include <mapreduce/yt/common/config.h>
#include <mapreduce/yt/interface/client.h>
#include <mapreduce/yt/util/ypath_join.h>

#include <util/generic/utility.h>

using namespace NCrypta;
using namespace NCrypta::NLookalike;
using namespace NCrypta::NLookalike::NUserSegmentsExporter;

namespace {
    TExportUserSegmentsReducer::TSegmentMetas ReadSegmentParentMetas(NYT::IClientBasePtr client, const TString& path) {
        TExportUserSegmentsReducer::TSegmentMetas result;

        for (auto reader = client->CreateTableReader<TSegmentMetaEntry>(path); reader->IsValid(); reader->Next()) {
            auto row = reader->MoveRow();

            result[row.GetSegmentId()].Swap(row.MutableMeta());
        }

        return result;
    }

    TString GetWorkingDir(NYT::IClientBasePtr client, const TExportUserSegmentsJobConfig& config) {
        if (config.GetMode() == ModeValue::NEW) {
            return NDirectoryFinder::FindLastWithChildren(client, config.GetVersionsDir(), {
                TYtNodeNames().GetFreshUserSegmentsDir(),
                TYtNodeNames().GetFreshMetasDir()
            });
        }

        return NDirectoryFinder::FindLastWithChildren(client, config.GetVersionsDir(), {
            TYtNodeNames().GetUserSegmentsTable(),
            TYtNodeNames().GetSegmentMetasTable()
        });
    }
}

int NUserSegmentsExporter::ExportUserSegmentsJob(TExportUserSegmentsJobConfig config, NLog::TLogPtr log) {
    NYT::TConfig::Get()->Pool = config.GetYt().GetPool();

    auto client = NYT::CreateClient(config.GetYt().GetProxy());
    auto tx = client->StartTransaction();

    log->info("Running in mode: {}", config.GetMode());

    TString workingDir = GetWorkingDir(tx, config);
    log->info("Found last version dir: {}", workingDir);

    const auto now = TShiftedClock::Now();

    TString usersSegmentsTable;
    TString segmentsMetasTable;
    TString freshTimestamp;
    TString versionExportPath;

    if (config.GetMode() == ModeValue::NEW) {
        auto maybeUserSegmentsPath = GetOldestTimedTable(tx, NYT::JoinYPaths(workingDir, TYtNodeNames().GetFreshUserSegmentsDir()));
        if (!maybeUserSegmentsPath) {
            tx->Commit();
            log->info("Fresh user segments table not found");
            return 0;
        }
        freshTimestamp = *maybeUserSegmentsPath;
        usersSegmentsTable = NYT::JoinYPaths(workingDir, TYtNodeNames().GetFreshUserSegmentsDir(), freshTimestamp);
        segmentsMetasTable = NYT::JoinYPaths(workingDir, TYtNodeNames().GetFreshMetasDir(), freshTimestamp);
        versionExportPath = NYT::JoinYPaths(workingDir, TYtNodeNames().GetFreshLalExportTable());

    } else {
        usersSegmentsTable = NYT::JoinYPaths(workingDir, TYtNodeNames().GetUserSegmentsTable());
        segmentsMetasTable = NYT::JoinYPaths(workingDir, TYtNodeNames().GetSegmentMetasTable());
        versionExportPath = NYT::JoinYPaths(workingDir, TYtNodeNames().GetLalExportTable());
    }


    auto segmentParentMetas = ReadSegmentParentMetas(tx, segmentsMetasTable);

    auto spec = NYT::TMapReduceOperationSpec()
                    .ReduceBy({"Shard"})
                    .ReducerSpec(NYT::TUserJobSpec().MemoryLimit(config.GetReducerMemoryLimitGb() * 1024 * 1024 * 1024));
    spec.AddInput<TUserSegments>(usersSegmentsTable);

    TTimedYtPathGenerator timedPathGenerator(now);
    const auto& exportPath = timedPathGenerator.GetPath(config.GetExportDir(), to_lower(ModeValue_Name(config.GetMode())) + "-{ts}-lookalike");
    const auto& exportLogPath = timedPathGenerator.GetPath(config.GetExportLogDir(), to_lower(ModeValue_Name(config.GetMode())) + "-{ts}-lookalike");
    const auto& errorsPath = timedPathGenerator.GetPath(config.GetErrorsDir());

    const auto& outputTableIndexes = TExportUserSegmentsReducer::PrepareOutput(spec, versionExportPath, errorsPath);

    tx->Remove(versionExportPath, NYT::TRemoveOptions().Force(true));

    log->info("Starting MapReduce operation with mapper: TExportUserSegmentsMapper, reducer: TExportUserSegmentsReducer");
    tx->MapReduce(
        spec,
        new TExportUserSegmentsMapper(config),
        new TExportUserSegmentsReducer(
            outputTableIndexes,
            now,
            config.GetShardsCount(),
            config.GetDesiredSegmentSize(),
            std::move(segmentParentMetas)));

    const TVector<TString> sortBy({"SegmentType", "SegmentID"});
    tx->Sort(versionExportPath, NYT::TRichYPath(versionExportPath).Schema(NYT::CreateTableSchema<TAudienceUserSegment>(sortBy)).OptimizeFor(NYT::EOptimizeForAttr::OF_SCAN_ATTR), sortBy);

    tx->Copy(versionExportPath, exportPath, NYT::TCopyOptions().Recursive(true));
    tx->Copy(versionExportPath, exportLogPath, NYT::TCopyOptions().Recursive(true));

    if (config.GetMode() == ModeValue::NEW) {
        log->info("Removing processed user segments table: {}", usersSegmentsTable);
        tx->Remove(usersSegmentsTable);
        log->info("Removing processed metas table: {}", segmentsMetasTable);
        tx->Remove(segmentsMetasTable);
    }

    tx->Commit();

    SetTtl(client, errorsPath, TDuration::Days(config.GetErrorsTtlDays()), ESetTtlMode::RemoveIfEmpty);
    SetTtl(client, exportLogPath, TDuration::Days(config.GetExportLogTtlDays()), ESetTtlMode::RemoveIfEmpty);

    return 0;
}
