#include "export_user_segments_mapper.h"

#include <crypta/lib/native/time/shifted_clock.h>
#include <crypta/lookalike/lib/native/segment_id_validator.h>
#include <crypta/lookalike/services/user_segments_exporter/proto/error.pb.h>

#include <library/cpp/iterator/zip.h>

#include <util/digest/murmur.h>
#include <util/random/shuffle.h>

using namespace NCrypta;
using namespace NCrypta::NLookalike;
using namespace NCrypta::NLookalike::NUserSegmentsExporter;

TExportUserSegmentsMapper::TExportUserSegmentsMapper(TExportUserSegmentsJobConfig config)
    : Config(config)
{
}

void TExportUserSegmentsMapper::Do(TReader* reader, TWriter* writer) {
    for (; reader->IsValid(); reader->Next()) {
        auto row = reader->MoveRow();

        const auto userId = row.GetUserId();
        if (Config.GetUseRandomShuffle()) {
            Shuffle(row);
        }

        TShardedUserSegment shardedUserSegment;
        shardedUserSegment.SetShard(MurmurHash<ui64>(&userId, sizeof(userId)) % Config.GetShardsCount());
        shardedUserSegment.MutableUserSegments()->Swap(&row);

        writer->AddRow(shardedUserSegment);
    }
}

void TExportUserSegmentsMapper::Shuffle(TUserSegments& row) {
    TVector<std::pair<ui64, float>> segmentScorePairs;
    TVector<ui64> segments;
    TVector<float> scores;

    for (const auto& [segmentId, score]: Zip(row.GetSegments(), row.GetScores())) {
        segmentScorePairs.push_back({segmentId, score});
    }

    ShuffleRange(segmentScorePairs);
    segmentScorePairs.resize(std::min(segmentScorePairs.size(), Config.GetMaxSegmentsPerUser()));


    for (const auto& [segmentId, score]: segmentScorePairs) {
        segments.push_back(segmentId);
        scores.push_back(score);
    }

    *row.MutableSegments() = {segments.begin(), segments.end()};
    *row.MutableScores() = {scores.begin(), scores.end()};
}

REGISTER_MAPPER(TExportUserSegmentsMapper);
