#include "export_user_segments_reducer.h"

#include "scored_user.h"

#include <crypta/lookalike/lib/native/segment_id_validator.h>
#include <crypta/lookalike/proto/audience_user_segment.pb.h>
#include <crypta/lookalike/services/user_segments_exporter/proto/error.pb.h>

#include <library/cpp/iterator/zip.h>

#include <util/generic/algorithm.h>
#include <util/generic/utility.h>

using namespace NCrypta;
using namespace NCrypta::NLookalike;
using namespace NCrypta::NLookalike::NUserSegmentsExporter;

TExportUserSegmentsReducer::TExportUserSegmentsReducer(
        TOutputIndexes outputIndexes,
        TInstant exportTs,
        size_t shardsCount,
        TInterval desiredSegmentSize,
        TSegmentMetas segmentMetas)
    : OutputIndexes(std::move(outputIndexes))
    , ExportTs(exportTs)
    , ShardsCount(shardsCount)
    , DesiredSegmentSize(desiredSegmentSize)
    , SegmentMetas(std::move(segmentMetas))
{
}

void TExportUserSegmentsReducer::Do(TReader* reader, TWriter* writer) {
    THashMap<ui64, TVector<TScoredUser>> segmentScoredUsers;

    for (; reader->IsValid(); reader->Next()) {
        const auto& row = reader->GetRow();
        const auto& userSegments = row.GetUserSegments();

        for (const auto& [segmentId, score] : Zip(userSegments.GetSegments(), userSegments.GetScores())) {
            segmentScoredUsers[segmentId].push_back({.UserId = userSegments.GetUserId(), .Score = score});
        }
    }

    for (auto& [segmentId, scoredUsers] : segmentScoredUsers) {
        if (!NSegmentIdValidator::IsLookalike(segmentId)) {
            WriteError(writer, segmentId, "Segment id is not valid lookalike id");
            continue;
        }

        size_t segmentSize = GetDesiredSegmentSize(segmentId) / ShardsCount;

        if (segmentSize < scoredUsers.size()) {
            NthElement(
                scoredUsers.begin(),
                scoredUsers.begin() + segmentSize,
                scoredUsers.end(),
                [](const auto& lhs, const auto& rhs) { return lhs.Score > rhs.Score; });
        } else {
            segmentSize = scoredUsers.size();
        }

        for (size_t i = 0; i < segmentSize; ++i) {
            WriteUserSegment(writer, scoredUsers[i].UserId, segmentId);
        }
    }
}

size_t TExportUserSegmentsReducer::GetDesiredSegmentSize(ui64 segmentId) const {
    const auto it = SegmentMetas.find(segmentId);
    return (it == SegmentMetas.end()) ? 0 : ClampVal(it->second.GetSize() * 2, DesiredSegmentSize.GetMin(), DesiredSegmentSize.GetMax());
}

void TExportUserSegmentsReducer::WriteUserSegment(TWriter* writer, ui64 userId, ui64 segmentId) {
    TAudienceUserSegment audienceUserSegment;
    audienceUserSegment.SetYandexuid(userId);
    audienceUserSegment.SetSegment(segmentId);
    audienceUserSegment.SetTimestamp(ExportTs.Seconds());
    audienceUserSegment.SetSegmentType("lookalike");

    writer->AddRow(audienceUserSegment, *OutputIndexes[EOutputTables::Export]);
}

void TExportUserSegmentsReducer::WriteError(TWriter* writer, ui64 segmentId, const TString& message) {
    TError error;
    error.SetSegment(segmentId);
    error.SetMessage(message);

    writer->AddRow(error, *OutputIndexes[EOutputTables::Errors]);
}

TExportUserSegmentsReducer::TOutputIndexes TExportUserSegmentsReducer::PrepareOutput(NYT::TOperationOutputSpecBase& spec, const TString& exportTable, const TString& errorsTable) {
    TOutputIndexes::TBuilder outputBuilder;

    outputBuilder.AddOutput<TAudienceUserSegment>(spec, exportTable, EOutputTables::Export);
    outputBuilder.AddOutput<TError>(spec, errorsTable, EOutputTables::Errors);

    return outputBuilder.GetIndexes();
}

REGISTER_REDUCER(TExportUserSegmentsReducer);
