#include "parse_visit_log_job.h"

#include "fields.h"
#include "parse_visit_log_mapper.h"

#include <crypta/lib/native/proto_secrets/remove_secrets.h>
#include <crypta/lib/native/range/packs.h>
#include <crypta/lib/native/yt/processed_tables_tracker/processed_tables_tracker.h>
#include <crypta/lib/native/yt/utils/helpers.h>
#include <crypta/lib/native/yt/utils/timed_yt_path_generator.h>
#include <crypta/lookalike/proto/counter_visit.pb.h>
#include <crypta/lookalike/proto/error.pb.h>
#include <crypta/lookalike/proto/goal_achievement.pb.h>

#include <mapreduce/yt/common/config.h>
#include <mapreduce/yt/interface/common.h>

#include <util/string/join.h>

using namespace NCrypta;
using namespace NCrypta::NLookalike;

int NVisitLogParser::ParseVisitLogJob(TParseVisitLogJobConfig config, NLog::TLogPtr log) {
    NYT::TConfig::Get()->Pool = config.GetYt().GetPool();

    auto client = NYT::CreateClient(config.GetYt().GetProxy());

    TProcessedTablesTracker processedTablesTracker(config.GetSource());
    const auto& sourceTables = processedTablesTracker.GetUnprocessedTables(client);

    log->info("List of source tables [\n{}\n]", JoinSeq(",\n", sourceTables));

    for (const auto& pack : GetPacks(sourceTables, config.GetTablePackSize())) {
        log->info("Processing pack [\n{}\n]", JoinSeq(",\n", pack));

        auto tx = client->StartTransaction();

        TTimedYtPathGenerator timedPathGenerator(TShiftedClock::Now());
        const TString tableNameFormat = "{ts}-" + config.GetLog();

        const auto& errorsTable = timedPathGenerator.GetPath(config.GetErrorsDir(), tableNameFormat);
        const auto& counterVisitsTable = timedPathGenerator.GetPath(config.GetCounterVisitsDir(), tableNameFormat);
        const auto& goalAchievementsTable = timedPathGenerator.GetPath(config.GetGoalAchievementsDir(), tableNameFormat);

        auto spec = NYT::TMapOperationSpec();
        AddInputs<NYT::TNode>(spec, GetTablesWithColumns(pack, {
            NFields::USER_ID,
            NFields::USER_ID_TYPE,
            NFields::COUNTER_ID,
            NFields::UTC_START_TIME,
            NFields::GOALS_ID,
            NFields::GOALS_EVENT_TIME
        }));

        TParseVisitLogMapper::TOutputIndexes::TBuilder outputBuilder;
        outputBuilder.AddOutput<TCounterVisit>(spec, counterVisitsTable, TParseVisitLogMapper::EOutputTables::CounterVisits);
        outputBuilder.AddOutput<TGoalAchievement>(spec, goalAchievementsTable, TParseVisitLogMapper::EOutputTables::GoalAchievements);
        outputBuilder.AddOutput<TError>(spec, errorsTable, TParseVisitLogMapper::EOutputTables::Errors);

        tx->Map(spec,
                new TParseVisitLogMapper(outputBuilder.GetIndexes()),
                NYT::TOperationOptions().Spec(NYT::TNode()("data_weight_per_job", config.GetDataWeightPerJobGb() * 1024 * 1024 * 1024)));

        tx->Sort(counterVisitsTable, counterVisitsTable, {YT_FIELD(TCounterVisit, CounterId), YT_FIELD(TCounterVisit, Ts)});
        tx->Sort(goalAchievementsTable, goalAchievementsTable, {YT_FIELD(TGoalAchievement, GoalId), YT_FIELD(TGoalAchievement, Ts)});

        processedTablesTracker.AddProcessedTables(tx, pack);

        tx->Commit();

        SetTtl(client, errorsTable, TDuration::Days(config.GetErrorsTtlDays()), ESetTtlMode::RemoveIfEmpty);
    }

    return 0;
}
