#include "parse_visit_log_mapper.h"

#include "fields.h"

#include <crypta/lib/native/retargeting_ids/retargeting_id.h>
#include <crypta/lookalike/proto/counter_visit.pb.h>
#include <crypta/lookalike/proto/error.pb.h>
#include <crypta/lookalike/proto/goal_achievement.pb.h>

#include <library/cpp/iterator/zip.h>

using namespace NCrypta;
using namespace NCrypta::NLookalike;
using namespace NCrypta::NLookalike::NVisitLogParser;

TParseVisitLogMapper::TParseVisitLogMapper(TOutputIndexes outputIndexes)
    : OutputIndexes(std::move(outputIndexes))
{
}

void TParseVisitLogMapper::Do(TReader* reader, TWriter* writer) {
    for (; reader->IsValid(); reader->Next()) {
        try {
            const auto& row = reader->GetRow();

            if (row.ChildAsUint64(NFields::USER_ID_TYPE) != 1) {
                continue;
            }

            const auto yandexuid = row.ChildAsUint64(NFields::USER_ID);

            ParseCounterVisit(writer, yandexuid, row);
            ParseGoalAchievements(writer, yandexuid, row);
        } catch (const yexception& e) {
            WriteError(writer, e.what());
        }
    }
}

void TParseVisitLogMapper::ParseCounterVisit(TWriter* writer, ui64 yandexuid, const NYT::TNode& node) {
    const auto counterId = node.ChildAsUint64(NFields::COUNTER_ID);
    const auto counterTs = node.ChildAsUint64(NFields::UTC_START_TIME);

    WriteCounterVisit(writer, yandexuid, counterId, counterTs);
}

void TParseVisitLogMapper::ParseGoalAchievements(TWriter* writer, ui64 yandexuid, const NYT::TNode& node) {
    const auto& nodeMap = node.AsMap();
    const auto& goalIdsIt = nodeMap.find(NFields::GOALS_ID);
    const auto& goalTimestampsIt = nodeMap.find(NFields::GOALS_EVENT_TIME);

    if ((goalIdsIt == nodeMap.end()) || (goalTimestampsIt == nodeMap.end()) || goalIdsIt->second.IsNull() || goalTimestampsIt->second.IsNull()) {
        return;
    }

    const auto& goalIds = goalIdsIt->second.AsList();
    const auto& goalTimestamps = goalTimestampsIt->second.AsList();

    Y_ENSURE(goalIds.size() == goalTimestamps.size(), "Goal ids count doesn't match goal timestamps count");

    for (const auto& [goalIdNode, goalTsNode] : Zip(goalIds, goalTimestamps)) {
        const auto goalId = goalIdNode.AsUint64();

        if (NRetargetingIds::TRetargetingId(goalId).GetType() != NRetargetingIds::EType::MetrikaGoal) {
            continue;
        }

        WriteGoalAchievement(writer, yandexuid, goalId, goalTsNode.AsUint64());
    }
}

void TParseVisitLogMapper::WriteCounterVisit(TWriter* writer, ui64 yandexuid, ui64 counterId, ui64 ts) {
    TCounterVisit counterVisit;
    counterVisit.SetCounterId(counterId);
    counterVisit.SetYandexuid(yandexuid);
    counterVisit.SetTs(-static_cast<i64>(ts));

    writer->AddRow(counterVisit, *OutputIndexes[EOutputTables::CounterVisits]);
}

void TParseVisitLogMapper::WriteGoalAchievement(TWriter* writer, ui64 yandexuid, ui64 goalId, ui64 ts) {
    TGoalAchievement goalAchievement;
    goalAchievement.SetGoalId(goalId);
    goalAchievement.SetYandexuid(yandexuid);
    goalAchievement.SetTs(-static_cast<i64>(ts));

    writer->AddRow(goalAchievement, *OutputIndexes[EOutputTables::GoalAchievements]);
}

void TParseVisitLogMapper::WriteError(TWriter* writer, const TString& message) {
    TError error;
    error.SetMessage(message);

    writer->AddRow(error, *OutputIndexes[EOutputTables::Errors]);
}

REGISTER_MAPPER(TParseVisitLogMapper);
