import datetime
import os

from library.python import resource
import yaml

from crypta.lib.python.data_size import DataSize
from crypta.lib.python.solomon.proto import alert_pb2
from crypta.lib.python.spine import event_processing_utils
from crypta.lib.python.spine.consts import environment
from crypta.lib.python.spine.consts.yt_proxy import YtProxy
from crypta.lib.python.spine.juggler import (
    consts,
    juggler_check_generator,
)
from crypta.lib.python.spine.juggler.aggregate_hosts import AggregateHosts
from crypta.lib.python.spine.juggler.flap_detector_params import FlapDetectorParams
from crypta.lib.python.spine.sandbox import sandbox_scheduler
from crypta.lib.python.spine.solomon import solomon_check_generator
from crypta.lib.python.spine.yt import yt_config_registry
from crypta.lib.python.spine.yt.yt_latency_metric import YtLatencyMetric
from crypta.lib.python.spine.yt.yt_size_metric import YtSizeMetric
from sandbox.projects.crypta.lookalike import (
    calc_goal_audiences,
    calc_metrika_counter_audiences,
    calc_metrika_segments,
    user_dssm_applier,
    visit_log_parser,
)
import sandbox.projects.crypta.lookalike.index_builder as index_builder
import sandbox.projects.crypta.lookalike.lal_refresher as lal_refresher
import sandbox.projects.crypta.lookalike.lal_synchronizer as lal_synchronizer
import sandbox.projects.crypta.lookalike.metrics.counts_per_parent_type_calc as counts_per_parent_type_calc
import sandbox.projects.crypta.lookalike.model_supplier as model_supplier
import sandbox.projects.crypta.lookalike.segment_dssm_applier as segment_dssm_applier
import sandbox.projects.crypta.lookalike.segmentator as segmentator
import sandbox.projects.crypta.lookalike.user_segments_exporter as user_segments_exporter


COMMON_AGGR_HOSTS = AggregateHosts(testing_host="crypta_lookalike_testing", production_host="crypta_lookalike")
LAL_MANAGER_NANNY_SERVICES = {
    "direct": {
        environment.TESTING: ("test_crypta_lal_lal_mngr_f_sas", "test_crypta_lal_lal_mngr_s_sas"),
        environment.PRODUCTION: ("prod_crypta_lal_lal_mngr_f_sas", "prod_crypta_lal_lal_mngr_s_sas"),
    },
}
DC = "sas"


def add_yt_size(yt_registry, path, threshold):
    YtSizeMetric(yt_registry, path).add_disk_space_alert(
        predicate=alert_pb2.GT,
        threshold=threshold,
    )


def add_yt_latency(yt_registry, path, threshold):
    YtLatencyMetric(yt_registry, path).add_latency_alert(
        threshold=threshold,
    )


def get_registry():
    index = yaml.safe_load(resource.find("/lal_index"))
    base_juggler_gen = juggler_check_generator.CryptaYtCheckGenerator(tags=["crypta-lookalike"])
    yt_registry = yt_config_registry.CryptaYtConfigRegistry(base_juggler_gen.clone(host=COMMON_AGGR_HOSTS.production_host.host))

    common_yt_sizes_paths = [
        ("errors/user_dssm_applier", DataSize(mb=0)),
        ("errors/visit_log_parser", DataSize(mb=0)),
        ("metrika_segments/errors", DataSize(mb=0)),
    ]

    scope_yt_sizes_paths = [
        ("errors/lal_refresher",  DataSize(mb=0)),
        ("errors/lal_synchronizer", DataSize(mb=0)),
        ("errors/segment_dssm_applier", DataSize(mb=0)),
        ("errors/segmentator", DataSize(mb=0)),
        ("errors/user_segments_exporter", DataSize(mb=0)),
    ]

    common_yt_latencies_paths = [
        ("goal_audiences/fresh", datetime.timedelta(hours=4)),
        ("metrika_counter_audiences/fresh", datetime.timedelta(hours=4)),
    ]

    common_prefix = "lookalike"

    for path, threshold in common_yt_sizes_paths:
        add_yt_size(yt_registry, os.path.join(common_prefix, path), threshold)

    for path, threshold in common_yt_latencies_paths:
        add_yt_latency(yt_registry, os.path.join(common_prefix, path), threshold)

    for context in index:
        scope = context["scope"]

        path_prefix = os.path.join(common_prefix, "scopes", scope)
        for path, threshold in scope_yt_sizes_paths:
            add_yt_size(yt_registry, os.path.join(path_prefix, path), threshold)

        solomon = solomon_check_generator.ServiceSolomonCheckGenerator(
            base_juggler_gen.clone(
                child_group="crypta_lookalike_lal_manager",
                child_group_type=consts.GroupType.host,
            ),
            project="crypta_lookalike",
            service="lal_manager",
            cluster=environment.PRODUCTION,
        )
        for logname, threshold, escalation in [
            (context["update_lal_slow_log"], 700, False),
            (context["update_lal_fast_log"], 600, True),
        ]:
            solomon.create_logbroker_lag_check(
                consumer="crypta/prod/lookalike/consumer",
                topic="crypta/prod/lookalike/{}".format(logname),
                threshold=threshold,
                period=datetime.timedelta(minutes=5),
            ).add_flap_detector(FlapDetectorParams(
                datetime.timedelta(minutes=15),
                datetime.timedelta(minutes=30),
            )).set_phone_escalation(escalation)

        for env, services in LAL_MANAGER_NANNY_SERVICES[scope].iteritems():
            for service in services:
                juggler = base_juggler_gen.clone(
                    child_group=service,
                    child_group_type=consts.GroupType.nanny,
                    warn_limit=0,
                    crit_limit=0,
                )
                juggler.icmpping().set_crit_limit(1).set_phone_escalation(escalation)

                solomon = solomon_check_generator.DcSolomonCheckGenerator(
                    juggler,
                    project="crypta_lookalike",
                    service="lal_manager",
                    cluster=env,
                    dc=DC,
                    # TODO(CRYPTAYT-3737) Add additional labels for slow/fast logs
                    additional_selectors={
                        "scope": scope
                    },
                )
                event_processing_utils.get_invalid_events_check(solomon)

                # Escalate if there are a lot of errors
                critical_errors_check = solomon.get_sensor("worker.processors.common.errors.critical").create_threshold_check(
                    aggregation=alert_pb2.SUM,
                    predicate=alert_pb2.GT,
                    threshold=3,
                    period=datetime.timedelta(hours=4),
                    description="Critical errors count: {{ pointValue }}",
                    juggler_service="worker.processors.common.errors.critical.escalation",
                ).add_nodata_mode(consts.NoDataMode.force_ok)

                if escalation:
                    critical_errors_check.add_phone_escalation()

                # Show in juggler for dutyman to check why there are errors and for adding cases to retriable errors
                solomon.get_sensor("worker.processors.common.errors.critical").create_threshold_check(
                    aggregation=alert_pb2.MAX,
                    predicate=alert_pb2.GT,
                    threshold=0,
                    period=datetime.timedelta(hours=24),
                    description="Critical errors count: {{ pointValue }}",
                ).add_nodata_mode(consts.NoDataMode.force_ok)

    add_sandbox_tasks(base_juggler_gen, index)

    return base_juggler_gen


class Task(object):
    def __init__(self, task, crit_time, aggr_hosts, schedule_interval, kill_timeout=datetime.timedelta(hours=3), retry_interval=None):
        self.task = task
        self.crit_time = crit_time
        self.aggr_hosts = aggr_hosts
        self.schedule_interval = schedule_interval
        self.kill_timeout = kill_timeout
        self.retry_interval = retry_interval


def add_sandbox_tasks(juggler, index):
    sandbox = sandbox_scheduler.create_default_generator(juggler.clone(yt_dependencies=YtProxy.Group.offline), ["LOOKALIKE"])

    prod_hosts = [COMMON_AGGR_HOSTS.production_host]
    all_hosts = list(COMMON_AGGR_HOSTS)

    common_sandbox_tasks = [
        Task(
            task=calc_goal_audiences.CryptaLookalikeCalcGoalAudiencesTask,
            crit_time=datetime.timedelta(hours=6),
            aggr_hosts=all_hosts,
            schedule_interval=datetime.timedelta(minutes=20),
            kill_timeout=datetime.timedelta(hours=8),
        ),
        Task(
            task=calc_metrika_counter_audiences.CryptaLookalikeCalcMetrikaCounterAudiencesTask,
            crit_time=datetime.timedelta(hours=4),
            aggr_hosts=all_hosts,
            schedule_interval=datetime.timedelta(minutes=30),
            kill_timeout=datetime.timedelta(hours=24),
        ),
        Task(
            task=calc_metrika_segments.CryptaLookalikeCalcMetrikaSegmentsTask,
            crit_time=datetime.timedelta(hours=4),
            aggr_hosts=all_hosts,
            schedule_interval=datetime.timedelta(hours=1),
        ),
        Task(
            task=user_dssm_applier.CryptaLookalikeUserDssmApplierTask,
            crit_time=datetime.timedelta(hours=4),
            aggr_hosts=all_hosts,
            schedule_interval=datetime.timedelta(minutes=10),
        ),
    ]

    scope_sandbox_tasks = [
        Task(
            task=counts_per_parent_type_calc.CryptaLookalikeCountsPerParentTypeCalcTask,
            crit_time=datetime.timedelta(hours=36),
            aggr_hosts=prod_hosts,
            schedule_interval=datetime.timedelta(hours=1),
        ),
        Task(
            task=lal_refresher.CryptaLookalikeLalRefresherTask,
            crit_time=datetime.timedelta(hours=4),
            aggr_hosts=all_hosts,
            schedule_interval=datetime.timedelta(hours=1),
        ),
        Task(
            task=lal_synchronizer.CryptaLookalikeLalSynchronizerTask,
            crit_time=datetime.timedelta(hours=2),
            aggr_hosts=all_hosts,
            schedule_interval=datetime.timedelta(minutes=15),
        ),
        Task(
            task=model_supplier.CryptaLookalikeModelSupplierTask,
            crit_time=datetime.timedelta(hours=4),
            aggr_hosts=all_hosts,
            schedule_interval=datetime.timedelta(minutes=15),
        ),
    ]

    mode_sandbox_tasks = [
        Task(
            task=index_builder.CryptaLookalikeIndexBuilderTask,
            crit_time=datetime.timedelta(hours=4),
            aggr_hosts=all_hosts,
            schedule_interval=datetime.timedelta(minutes=10),
        ),
        Task(
            task=segmentator.CryptaLookalikeSegmentatorTask,
            crit_time=datetime.timedelta(hours=12),
            aggr_hosts=all_hosts,
            schedule_interval=datetime.timedelta(minutes=10),
            kill_timeout=datetime.timedelta(hours=6),
        ),
    ]

    mode_tasks_dict = {
        "NEW": mode_sandbox_tasks + [
            Task(
                task=segment_dssm_applier.CryptaLookalikeSegmentDssmApplierTask,
                crit_time=datetime.timedelta(hours=4),
                aggr_hosts=all_hosts,
                schedule_interval=datetime.timedelta(minutes=50),
            ),
            Task(
                task=user_segments_exporter.CryptaLookalikeUserSegmentsExporterTask,
                crit_time=datetime.timedelta(hours=12),
                aggr_hosts=all_hosts,
                schedule_interval=datetime.timedelta(minutes=10),
                kill_timeout=datetime.timedelta(hours=10),
            ),
        ],
        "ALL": mode_sandbox_tasks + [
            Task(
                task=segment_dssm_applier.CryptaLookalikeSegmentDssmApplierTask,
                crit_time=datetime.timedelta(hours=28),
                aggr_hosts=all_hosts,
                schedule_interval=datetime.timedelta(hours=24),
                retry_interval=datetime.timedelta(minutes=15),
            ),
            Task(
                task=user_segments_exporter.CryptaLookalikeUserSegmentsExporterTask,
                crit_time=datetime.timedelta(hours=12),
                aggr_hosts=all_hosts,
                schedule_interval=datetime.timedelta(hours=1),
                kill_timeout=datetime.timedelta(hours=10),
            ),
        ],
    }

    for task in common_sandbox_tasks:
        for aggr_host in task.aggr_hosts:
            sandbox.create_scheduler(
                task.task,
                schedule_interval=task.schedule_interval,
                env=aggr_host.environment,
                kill_timeout=task.kill_timeout,
            ).check(task.crit_time).set_host(aggr_host.host)

    for context in index:
        scope = context["scope"]

        for task in scope_sandbox_tasks:
            for aggr_host in task.aggr_hosts:
                sandbox.create_scheduler(
                    task.task,
                    schedule_interval=task.schedule_interval,
                    env=aggr_host.environment,
                    kill_timeout=task.kill_timeout,
                    retry_interval=task.retry_interval,
                    extra_params={"scope": scope},
                ).check(
                    task.crit_time,
                    juggler_service=task.task.get_juggler_service_for_scope(scope),
                ).set_host(
                    aggr_host.host,
                )

        for mode in ["ALL", "NEW"]:
            for task in mode_tasks_dict[mode]:
                for aggr_host in task.aggr_hosts:
                    sandbox.create_scheduler(
                        task.task,
                        schedule_interval=task.schedule_interval,
                        env=aggr_host.environment,
                        kill_timeout=task.kill_timeout,
                        extra_params={"scope": scope, "mode": mode},
                    ).check(
                        task.crit_time,
                        juggler_service=task.task.get_juggler_service_for_scope_and_mode(scope, mode),
                    ).set_host(
                        aggr_host.host,
                    )

    for aggr_host in all_hosts:
        for log in ["visit-v2-log", "visit-v2-private-log"]:
            sandbox.create_scheduler(
                visit_log_parser.CryptaLookalikeVisitLogParserTask,
                schedule_interval=datetime.timedelta(minutes=30),
                env=aggr_host.environment,
                extra_params={"log": log},
                retry_interval=datetime.timedelta(minutes=1),
            ).check(
                datetime.timedelta(hours=4),
                juggler_service=visit_log_parser.CryptaLookalikeVisitLogParserTask.get_juggler_service_for_log(log),
            ).set_host(
                aggr_host.host
            )
