package ru.yandex.crypta.service.ltp.client;

import java.time.Duration;
import java.util.Optional;

import javax.inject.Inject;

import io.grpc.ManagedChannelBuilder;
import io.grpc.StatusRuntimeException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.crypta.lab.utils.GrpcRetries;
import ru.yandex.crypta.lib.proto.TLtpViewerConfig;
import ru.yandex.crypta.ltp.viewer.proto.TPreloadHistoryCommand;
import ru.yandex.crypta.ltp.viewer.services.api.proto.LtpViewerGrpc;
import ru.yandex.crypta.ltp.viewer.services.api.proto.TGetHistoryRequest;
import ru.yandex.crypta.ltp.viewer.services.api.proto.TGetHistoryResponse;
import ru.yandex.crypta.ltp.viewer.services.api.proto.TGetProgressRequest;
import ru.yandex.crypta.ltp.viewer.services.api.proto.TGetProgressResponse;
import ru.yandex.crypta.ltp.viewer.services.api.proto.TGetUserQueriesRequest;
import ru.yandex.crypta.ltp.viewer.services.api.proto.TGetUserQueriesResponse;
import ru.yandex.crypta.ltp.viewer.services.api.proto.TPreloadHistoryResponse;

public class LtpViewerClient {
    private final LtpViewerGrpc.LtpViewerBlockingStub blockingStub;
    private static final Logger LOG = LoggerFactory.getLogger(ru.yandex.crypta.service.ltp.client.LtpViewerClient.class);

    @Inject
    public LtpViewerClient(TLtpViewerConfig config) {
        var channel = ManagedChannelBuilder.forAddress(config.getHost(), config.getPort())
                .defaultServiceConfig(
                        GrpcRetries.buildConfig(4D, Duration.ofMillis(50), Duration.ofSeconds(1), 2D)
                )
                .enableRetry()
                .maxInboundMessageSize(100 * 1000 * 1000)
                .usePlaintext()
                .build();
        blockingStub = LtpViewerGrpc.newBlockingStub(channel);
    }

    public Optional<TGetHistoryResponse> getHistory(TGetHistoryRequest request) {
        try {
            return Optional.of(blockingStub.getHistory(request));
        } catch (StatusRuntimeException e) {
            LOG.warn("RPC failed: {}", e.getStatus());
            return Optional.empty();
        }
    }

    public Optional<TPreloadHistoryResponse> preloadHistory(String idType, String id, String owner, String fromDate, String toDate) {
        try {
            var command = TPreloadHistoryCommand.newBuilder();
            command.getIdBuilder().setType(idType).setValue(id);
            command.setOwner(owner)
                   .setFromDate(fromDate == null ? "" : fromDate)
                   .setToDate(toDate == null ? "" : toDate);
            return Optional.of(blockingStub.preloadHistory(command.build()));
        } catch (StatusRuntimeException e) {
            LOG.warn("RPC failed: {}", e.getStatus());
            return Optional.empty();
        }
    }

    public Optional<TGetUserQueriesResponse> getUserQueries(String login) {
        try {
            return Optional.of(blockingStub.getUserQueries(TGetUserQueriesRequest.newBuilder().setOwner(login).build()));
        } catch (StatusRuntimeException e) {
            LOG.warn("RPC failed: {}", e.getStatus());
            return Optional.empty();
        }
    }

    public Optional<TGetProgressResponse> getProgress(String idType, String id, String fromDate, String toDate) {
        try {
            var request = TGetProgressRequest.newBuilder();
            request.getIdBuilder().setType(idType).setValue(id);
            request.setFromDate(fromDate).setToDate(toDate);
            return Optional.of(blockingStub.getProgress(request.build()));
        } catch (StatusRuntimeException e) {
            LOG.warn("RPC failed: {}", e.getStatus());
            return Optional.empty();
        }
    }
}
