from datetime import timedelta

from crypta.lib.python import crypta_env
from crypta.lib.python.data_size import DataSize
from crypta.lib.python.solomon.proto import alert_pb2
from crypta.lib.python.spine import (
    arcadia_ci,
    deploy,
)
from crypta.lib.python.spine.consts import (
    environment,
    dc as dc_consts,
)
from crypta.lib.python.spine.consts.yt_proxy import YtProxy
import crypta.lib.python.spine.deploy.consts as yp_consts
from crypta.lib.python.spine.juggler import juggler_check_generator
from crypta.lib.python.spine.juggler.flap_detector_params import FlapDetectorParams
from crypta.lib.python.spine.sandbox import sandbox_scheduler
from crypta.lib.python.spine.solomon import solomon_check_generator


GRPC_PORT = 80
HTTP_PORT = 81
SOLOMON_PORT = 4242
BALANCER = "ltp-viewer.crypta.yandex.net"


def get_registry():
    juggler = juggler_check_generator.CryptaYtCheckGenerator(
        host="crypta-ltp",
        tags=["crypta-ltp"],
    )

    release_registry = arcadia_ci.CryptaReleaseGenerator(juggler).standard_release(
        title="LTP Viewer",
        abs_paths=["crypta/ltp/**"],
    )

    add_ltp_viewer_api(juggler, release_registry)
    add_ltp_viewer_worker(juggler, release_registry)
    add_ltp_viewer_index(juggler, release_registry)
    return juggler


def add_ltp_viewer_api(base_juggler_check_generator, release_registry):
    release_registry.add_docker_image("crypta/ltp/viewer/services/api/docker/crypta-ltp-viewer-api.json")

    deploy_gen = base_juggler_check_generator.add_subregistry(deploy.CryptaDeployStageGenerator())
    stage = deploy_gen.stage("crypta-ltp-viewer-api")
    deploy_unit = stage.multi_cluster_deploy_unit(
        pods_per_cluster={dc_consts.SAS: 2},
        cpu_ms=500,
        ram=DataSize(gb=1),
        storage=deploy.HDD(
            capacity=DataSize(gb=5),
            bandwidth_guarantee=DataSize(mb=10),
            bandwidth_limit=DataSize(mb=10),
        ),
        project_network=yp_consts.CRYPTANETS,
        network_bandwidth=DataSize(mb=1),
        id_=environment.PRODUCTION,
    ).add_endpoint_set(
        port=GRPC_PORT,
        id_="grpc",
    ).add_endpoint_set(
        port=HTTP_PORT,
        id_="http",
    ).add_balancer(
        BALANCER,
    ).deployment_strategy(
        max_unavailable=1,
    )

    box = deploy_unit.box(
        docker_image="crypta/crypta-ltp-viewer-api",
    ).docker_release_rule(
        environment.STABLE,
    ).add_init_command(
        deploy.Command(
            "/root/init.sh"
        ),
    ).literal_env(
        crypta_env.Production.crypta_environment
    ).literal_env(
        crypta_env.LiteralEnv("GRPC_PORT", str(GRPC_PORT))
    ).literal_env(
        crypta_env.LiteralEnv("HTTP_PORT", str(HTTP_PORT))
    ).secret_env(
        # TODO(cherenkov-p-a) add metrics
        crypta_env.solomon_token
    ).secret_env(
        crypta_env.Production.ltp_viewer_tvm_secret
    ).secret_env(
        crypta_env.Production.ydb_token
    )

    grpc_cmd = deploy.Command(
        "/root/grpcurl -plaintext localhost:{} NCrypta.NLtpViewer.LtpViewer.Ping".format(GRPC_PORT)
    )

    box.workload(
        start_cmd="/root/api.sh",
        id_="grpc",
    ).liveness_cmd_check(
        grpc_cmd,
    ).readiness_cmd_check(
        grpc_cmd,
    )

    box.workload(
        start_cmd="/root/http_proxy.sh",
        id_="http",
    ).liveness_http_check(
        path="/ping",
        port=HTTP_PORT,
    ).readiness_http_check(
        path="/ping",
        port=HTTP_PORT,
    )

    box.workload(
        start_cmd="/root/expire_periodic.sh",
        id_="expire",
    )

    juggler = deploy_unit.get_juggler_generator(base_juggler_check_generator.clone(
        warn_limit=0,
        crit_limit=100,
        escalation=False,
        host="crypta-ltp-viewer-api",
    ))

    flap_detector_params = FlapDetectorParams(timedelta(minutes=5), timedelta(minutes=10))
    juggler.icmpping().add_flap_detector(flap_detector_params)
    juggler.http(
        "alive",
        path="/ping",
        port=HTTP_PORT,
    ).add_flap_detector(flap_detector_params)


def add_ltp_viewer_worker(base_juggler_check_generator, release_registry):
    release_registry.add_docker_image("crypta/ltp/viewer/services/worker/docker/crypta-ltp-viewer-worker.json")

    deploy_gen = base_juggler_check_generator.add_subregistry(deploy.CryptaDeployStageGenerator())
    stage = deploy_gen.stage("crypta-ltp-viewer-worker")
    deploy_unit = stage.multi_cluster_deploy_unit(
        pods_per_cluster={dc_consts.SAS: 2},
        cpu_ms=2000,
        ram=DataSize(gb=2),
        storage=deploy.HDD(
            capacity=DataSize(gb=5),
            bandwidth_guarantee=DataSize(mb=10),
            bandwidth_limit=DataSize(mb=10),
        ),
        project_network=yp_consts.CRYPTANETS,
        network_bandwidth=DataSize(mb=10),
        id_=environment.PRODUCTION,
    ).add_endpoint_set(
        port=SOLOMON_PORT,
        id_="solomon",
    ).deployment_strategy(
        max_unavailable=1,
    )

    box = deploy_unit.box(
        docker_image="crypta/crypta-ltp-viewer-worker",
    ).docker_release_rule(
        environment.STABLE,
    ).add_init_command(
        deploy.Command(
            "/root/init.sh"
        ),
    ).literal_env(
        crypta_env.Production.crypta_environment
    ).literal_env(
        crypta_env.LiteralEnv("LTP_VIEWER_API_ENDPOINT", "{}:{}".format(BALANCER, GRPC_PORT))
    ).secret_env(
        crypta_env.solomon_token
    ).secret_env(
        crypta_env.Production.ltp_viewer_tvm_secret
    ).secret_env(
        crypta_env.Production.yt_token
    ).secret_env(
        crypta_env.Production.ydb_token
    )

    box.workload(
        start_cmd="/root/worker.sh",
        id_="worker",
    )

    juggler = deploy_unit.get_juggler_generator(base_juggler_check_generator.clone(
        warn_limit=0,
        crit_limit=100,
        escalation=False,
        host="crypta-ltp-viewer-worker",
    ))

    flap_detector_params = FlapDetectorParams(timedelta(minutes=5), timedelta(minutes=10))
    juggler.icmpping().add_flap_detector(flap_detector_params)

    solomon = solomon_check_generator.DcSolomonCheckGenerator(
        juggler,
        project="crypta_ltp_viewer",
        service="metrics",
        cluster="worker",
        dc="sas",
    )
    for cmd_type in ("PreloadHistoryChunkCommand", "PreloadHistoryCommand"):
        solomon.get_sensor({"sensor": "errors", "cmd_type": cmd_type}).create_threshold_check(
            aggregation=alert_pb2.MAX,
            predicate=alert_pb2.GT,
            threshold=0,
            period=timedelta(hours=1),
            description="Errors: {{ pointValue }}",
        )

    solomon.create_logbroker_lag_check(
        consumer="crypta/prod/ltp/viewer-worker-consumer",
        topic="crypta/prod/ltp/viewer-task-queue",
        threshold=10**4,
        period=timedelta(minutes=5),
    )


def add_ltp_viewer_index(juggler, release_registry):
    bundle_id = release_registry.add_universal_bundle("crypta/ltp/viewer/services/build_index/crypta-ltp-viewer-build-index.json").bundle_id
    sandbox = sandbox_scheduler.create_default_generator(juggler, ["IDFY"])

    sandbox.create_run_universal_bundle_scheduler(
        description="Udpate LTP index",
        bundle_name=bundle_id,
        env=environment.STABLE,
        cmd=["{{cwd}}/main"],
        templates=[],
        schedule_daily_start_time="2021-12-28T06:00:00Z",
        kill_timeout=timedelta(hours=16),
        sequential_run=True,
        retry_interval=timedelta(minutes=15),
        additional_env={
            'ENV_TYPE': environment.PRODUCTION.upper(),
            'ENVIRONMENT': environment.PRODUCTION.lower(),
            'YT_PROXY': 'hahn.yt.yandex.net',
            'YT_POOL': 'crypta_graph'
        },
    ).check(
        crit_time=timedelta(days=2),
    ).add_yt_dependencies(
        YtProxy.Group.offline,
    )
