from cached_property import cached_property
from yt import yson

import urllib.parse


class Format:
    def format(self, record, context):
        raise NotImplementedError

    @cached_property
    def columns(self):
        raise NotImplementedError

    @cached_property
    def chyt_columns(self):
        return {name: name for name in self.columns}


class ColumnFormat(Format):
    def __init__(self, name):
        self.name = name

    @cached_property
    def columns(self):
        return [self.name]

    def format(self, record, context):
        value = record.get(self.name)
        if value is not None:
            yield from self.format_column(record.get(self.name), context)

    def format_column(self, value, context):
        raise NotImplementedError


class ColumnNamedFormat(ColumnFormat):
    def __init__(self, name, readable_name=None):
        self.name = name
        self.readable_name = readable_name or name

    def format_column(self, value, context):
        yield "{}: {}".format(self.readable_name, self.format_value(value, context))

    def format_value(self, value, context):
        return value


class YsonColumnNamedFormat(ColumnNamedFormat):
    @cached_property
    def chyt_columns(self):
        return {self.name: "ConvertYson({0}, 'text') AS {0}".format(self.name)}

    def format_value(self, value, context):
        return yson.loads(yson.get_bytes(value))


class URLColumnNamedFormat(ColumnNamedFormat):
    def format_value(self, value, context):
        return urllib.parse.unquote(value)


class GeoColumnNamedFormat(ColumnFormat):
    def __init__(self, name, readable_name, id_readable_name=None):
        self.name = name
        self.readable_name = readable_name
        self.id_readable_name = id_readable_name or name

    def format_column(self, value, context):
        yield "{}: {}".format(self.id_readable_name, value)
        yield "{}: {}".format(self.readable_name, context.geo.get_name(value))


class PageFormat(ColumnNamedFormat):
    def __init__(self, name, readable_name, readable_description):
        self.name = name
        self.readable_name = readable_name
        self.readable_description = readable_description

    def format_column(self, value, context):
        yield "{}: {}".format(self.name, value)
        yield "{}: {}".format(self.readable_name, context.pages.get_name(value))
        yield "{}: {}".format(self.readable_description, context.pages.get_description(value))


class CategoryFormat(ColumnNamedFormat):
    def format_value(self, value, context):
        return "{} {}".format(value, context.categories.get_description(value))


class CryptaTxDescription(Format):
    @cached_property
    def columns(self):
        return ["ItemQuantity", "ItemDescription"]

    def format(self, record, context):
        quantity = record.get("ItemQuantity")
        description = record.get("ItemDescription")
        if quantity is not None and description is not None:
            yield "Description: {} x {}".format(quantity, description)


class CryptaTxPriceFormat(ColumnNamedFormat):
    def format_value(self, value, context):
        return "{} RUB".format(value)


class LtpWatchAdditionalDescriptionFormat(Format):
    @cached_property
    def columns(self):
        return ["DeviceModel", "OSFamily", "BrowserName"]

    def format(self, record, context):
        yield "|".join(record[key] for key in self.columns if record.get(key))


class LtpRsyaClicksAdditionalDescriptionFormat(Format):
    @cached_property
    def columns(self):
        return ["ImpressionOptions", "Options", "DetailedDeviceType", "BrowserName"]

    def format(self, record, context):
        impression_options = record.get("ImpressionOptions", "")
        parts = []
        if "in-app" in impression_options:
            parts.append("inapp")
        if "is-ssp" in impression_options:
            parts.append("ssp")
        if "itp" in record.get("Options", ""):
            parts.append("itp")
        parts.extend(record[key] for key in ("DetailedDeviceType", "BrowserName") if record.get(key))
        yield "|".join(parts)
