import multiprocessing
import threading
import time

from crypta.lib.python.yt import yt_helpers
from crypta.lib.python.shared_data_structures.hot_swap_immutable_shareable_dict import HotSwapImmutableShareableDict


class TableProxy:
    def update(self, yt, path):
        self.do_update(yt.read_table(path))

    def do_update(self, rows):
        raise NotImplementedError


class Pages(TableProxy):
    def __init__(self, manager):
        self.names = HotSwapImmutableShareableDict("page_names", manager)
        self.descriptions = HotSwapImmutableShareableDict("page_descriptions", manager)

    def do_update(self, rows):
        names = {}
        descriptions = {}
        for page in rows:
            names[int(page["PageID"])] = str(page["Name"])
            descriptions[int(page["PageID"])] = str(page["Description"])
        self.names.update(names)
        self.descriptions.update(descriptions)

    def get_view(self):
        return PagesView(self)


class PagesView:
    def __init__(self, pages):
        self.names = pages.names.create_view()
        self.descriptions = pages.descriptions.create_view()

    def get_name(self, page_id):
        return self.names.get(page_id)

    def get_description(self, page_id):
        return self.descriptions.get(page_id)


class Categories(TableProxy):
    def __init__(self, manager):
        self.descriptions = HotSwapImmutableShareableDict("categories_descriptions", manager)

    def do_update(self, rows):
        self.descriptions.update({int(cat["BMCategoryID"]): str(cat["Description"]) for cat in rows})

    def get_view(self):
        return CategoriesView(self)


class CategoriesView:
    def __init__(self, categories):
        self.descriptions = categories.descriptions.create_view()

    def get_description(self, category_id):
        return self.descriptions.get(category_id)


class Regions(TableProxy):
    def __init__(self, manager):
        self.names = HotSwapImmutableShareableDict("region_names", manager)

    def do_update(self, rows):
        self.names.update({int(region["RegionID"]): str(region["Name"]) for region in rows if region["Lang"] == "ru"})

    def get_view(self):
        return RegionsView(self)


class RegionsView:
    def __init__(self, regions):
        self.names = regions.names.create_view()

    def get_name(self, region_id):
        return self.names.get(region_id)


class Context:
    def __init__(self):
        self.manager = multiprocessing.Manager()
        self.pages = Pages(self.manager)
        self.categories = Categories(self.manager)
        self.regions = Regions(self.manager)

    def update(self, yt, config):
        self.pages.update(yt, config.PagesPath)
        self.categories.update(yt, config.CategoriesPath)
        self.regions.update(yt, config.RegionsPath)

    def periodic_update(self, yt, config):
        while True:
            time.sleep(config.UpdatePeriodSec)
            self.update(yt, config)

    def start(self, config):
        yt = yt_helpers.get_yt_client(config.Yt.Proxy, config.Yt.Pool)

        self.update(yt, config.Context)

        self.thread = threading.Thread(target=self.periodic_update, args=(yt, config.Context))
        self.thread.daemon = True
        self.thread.start()


class ContextView:
    def __init__(self, context):
        self.pages = context.pages.get_view()
        self.categories = context.categories.get_view()
        self.geo = context.regions.get_view()
