from datetime import timedelta as dt

from crypta.lib.python.spine.consts import environment
from crypta.lib.python.spine.consts.yt_proxy import YtProxy
from crypta.lib.python.spine.juggler import (
    aggregate_hosts,
    consts,
    juggler_check_generator,
)
from crypta.lib.python.spine.sandbox import sandbox_scheduler
from crypta.ml.spine import spine_helpers
from crypta.profile.spine import profile_spine_helpers


PROD_HOST = "crypta.ml.production"

STABLE = aggregate_hosts.AggregateHost(PROD_HOST, environment.STABLE)
PRODUCTION = aggregate_hosts.AggregateHost(PROD_HOST, environment.PRODUCTION)
TESTING = aggregate_hosts.AggregateHost("crypta.ml.testing", environment.TESTING)

PROD_ONLY = [PRODUCTION]
ALL_HOSTS = [PRODUCTION, TESTING]
STABLE_ONLY = [STABLE]
ALL_SANDBOX_HOSTS = [STABLE, TESTING]


def get_registry():
    juggler = juggler_check_generator.JugglerCheckGenerator(refresh_time=90)

    def _hitman_check(service, host, monitoring_type, child_service_suffix, child_host_suffix, ttl):
        child_service = "{}.{}".format(monitoring_type, child_service_suffix)
        child_host = "hitman.process.{}".format(child_host_suffix)
        ttl = int(ttl.total_seconds())
        return juggler.any(service).set_child(
            group=child_host,
            group_type=consts.GroupType.host,
            service=child_service,
        ).set_host(host).set_ttl(ttl).add_nodata_mode(consts.NoDataMode.force_crit)

    for service, aggr_hosts, monitoring_type in [
        ("top_common_lal_apps", ALL_HOSTS, consts.Hitman.failed_jobs),
        ("train_lal_model", ALL_HOSTS, consts.Hitman.failed_jobs),
        ("train_rt_socdem_models", ALL_HOSTS, consts.Hitman.failed_jobs),
    ]:
        for aggr_host in aggr_hosts:
            child_host_suffix = "{}_{}".format(service, aggr_host.environment)
            child_service_suffix = "{}_{}".format(service, aggr_host.environment)
            _hitman_check(service, aggr_host.host, monitoring_type, child_service_suffix, child_host_suffix, dt(days=1))

    for service, aggr_host, monitoring_type, child_service_suffix, child_host_suffix, ttl in [
        ("affinitive_geo_orgs", PRODUCTION, consts.Hitman.failed_jobs, "affinitive_geo_orgs_production", "affinitive_geo_org_embeddings_production", dt(hours=27)),
        ("affinitive_geo_orgs_freshness", PRODUCTION, consts.Hitman.succeeded_jobs, "affinitive_geo_orgs_production", "affinitive_geo_org_embeddings_production", dt(hours=40)),
        ("prism_offline_weighting", PRODUCTION, consts.Hitman.failed_jobs, "prism_offline_weighting_production", "prism_offline_weighting_production", dt(hours=27)),
        ("prism_priors", PRODUCTION, consts.Hitman.failed_jobs, "prism_priors_production", "prism_priors_production", dt(hours=27)),
        ("prism_priors_freshness", PRODUCTION, consts.Hitman.succeeded_jobs, "prism_priors_production", "prism_priors_production", dt(hours=40)),
        ("rt_prism", PRODUCTION, consts.Hitman.failed_jobs, "train_rt_prism_model_production", "train_rt_prism_model_production", dt(days=8)),
        ("train_custom_segment_models", PRODUCTION, consts.Hitman.failed_jobs, "train_custom_segment_models", "train_crypta_custom_models", dt(hours=27)),
        ("train_mobile_socdem_model", PRODUCTION, consts.Hitman.failed_jobs, "train_mobile_socdem_model_production", "train_mobile_socdem_models", dt(hours=27)),
        ("train_mobile_socdem_model", TESTING, consts.Hitman.failed_jobs, "train_mobile_socdem_model_testing", "train_mobile_socdem_model_testing", dt(hours=27)),
        ("train_socdem_model", PRODUCTION, consts.Hitman.failed_jobs, "train_socdem_model_production", "train_socdem_models_production", dt(hours=27)),
        ("train_socdem_model", TESTING, consts.Hitman.failed_jobs, "train_socdem_model_testing", "train_socdem_model_testing", dt(hours=27)),
    ]:
        _hitman_check(service, aggr_host.host, monitoring_type, child_service_suffix, child_host_suffix, ttl)

    for service, aggr_hosts in [
        ("apps_clustering_metrics", STABLE_ONLY),
        ("import_socdem_data", PROD_ONLY),
        ("monitoring", PROD_ONLY),
        ("coded_segments", PROD_ONLY),
        ("sites_clustering_metrics", STABLE_ONLY),
        ("validate_socdem_profiles", PROD_ONLY),
    ]:
        for aggr_host in aggr_hosts:
            juggler.any(service).set_child(
                group="tag={} & host={}".format(service, aggr_host.host),
                group_type=consts.GroupType.events,
            ).set_host(
                aggr_host.host,
            ).set_ttl(
                int(dt(hours=32).total_seconds()),
            ).add_nodata_mode(
                consts.NoDataMode.force_crit,
            )

    for service, aggr_hosts, ttl in [
        ("lal_model_dict_update", PROD_ONLY, dt(hours=72)),
        ("priors_correlation", PROD_ONLY, dt(hours=32)),
        ("prism_correlation", PROD_ONLY, dt(hours=32)),
    ]:
        for aggr_host in aggr_hosts:
            juggler.any(service).set_child(
                service=service,
                group=aggr_host.host,
                group_type=consts.GroupType.host,
            ).set_host(
                aggr_host.host,
            ).set_ttl(
                int(ttl.total_seconds()),
            ).add_nodata_mode(
                consts.NoDataMode.force_crit,
            )

    for service, aggr_hosts, ttl in [
        ("outdated_coded_segments", PROD_ONLY, dt(hours=36)),
    ]:
        for aggr_host in aggr_hosts:
            juggler.direct(service).set_host(aggr_host.host).set_ttl(int(ttl.total_seconds()))

    _add_profile_schedulers(juggler)
    _add_siberia_schedulers(juggler)
    _add_lookalike_features_export_schedulers(juggler)
    _add_affinitive_geo_schedulers(juggler)

    return juggler


def _add_profile_schedulers(juggler):
    sandbox = sandbox_scheduler.create_default_generator(juggler.clone(yt_dependencies=YtProxy.Group.offline, tags=["crypta-profile"]), ["CRYPTA_PROFILE"])

    for task in [
        profile_spine_helpers.UniversalTask(
            bundle_name="profile-matching",
            binary="matching",
            schedule_daily_start_time="2018-12-04T10:30:00Z",
            kill_timeout=dt(hours=12),
            retry_interval=dt(hours=2),
            crit_time=dt(days=2),
            aggr_hosts=ALL_SANDBOX_HOSTS,
        ),
    ]:
        profile_spine_helpers.create_run_universal_bundle_schedulers(sandbox, task)

    for subprogram in ["update_app2vec", "update_site2vec"]:
        task = spine_helpers.UniversalTask(
            bundle_name="profile-vectors-update",
            binary="crypta-profile-vectors-update",
            subprogram=subprogram,
            additional_env={
                "CRYPTA_ENVIRONMENT": "production",
            },
            schedule_daily_start_time="2022-04-27T09:00:00Z",
            kill_timeout=dt(hours=24),
            retry_interval=dt(hours=2),
            crit_time=dt(hours=32),
            aggr_hosts=STABLE_ONLY,
        )
        spine_helpers.create_run_universal_bundle_schedulers(sandbox, task)


def _add_siberia_schedulers(juggler):
    sandbox = sandbox_scheduler.create_default_generator(juggler, ["SIBERIA"])

    for subprogram in ["update_clusters", "update_sites", "update_user_clusters"]:
        task = spine_helpers.UniversalTask(
            bundle_name="siberia-custom-audience-sites-clustering",
            binary="crypta-siberia-custom-audience-sites-clustering",
            subprogram=subprogram,
            additional_env={
                "CRYPTA_ENVIRONMENT": "production",
            },
            schedule_interval=dt(hours=1),
            kill_timeout=dt(hours=24),
            retry_interval=dt(hours=1),
            crit_time=dt(hours=32),
            aggr_hosts=STABLE_ONLY,
        )
        spine_helpers.create_run_universal_bundle_schedulers(sandbox, task)


def _add_lookalike_features_export_schedulers(juggler):
    sandbox = sandbox_scheduler.create_default_generator(juggler, ["LOOKALIKE"])

    for task in [
        spine_helpers.UniversalTask(
            bundle_name="lookalike-features-export",
            binary="crypta-lookalike-features-export",
            subprogram="compute_caesar_goal_features",
            additional_env={
                "CRYPTA_ENVIRONMENT": "production",
            },
            schedule_daily_start_time="2022-01-28T09:00:00Z",
            kill_timeout=dt(hours=24),
            retry_interval=dt(hours=2),
            crit_time=dt(hours=32),
            aggr_hosts=STABLE_ONLY,
        ),
    ]:
        spine_helpers.create_run_universal_bundle_schedulers(sandbox, task)


def _add_affinitive_geo_schedulers(juggler):
    sandbox = sandbox_scheduler.create_default_generator(juggler, ["AFFINITIVE_GEO"])

    for task in [
        spine_helpers.UniversalTask(
            bundle_name="affinitive-geo-orgvisits",
            binary="crypta-affinitive-geo-orgvisits",
            templates=["config.yaml"],
            additional_env={
                "CRYPTA_ENVIRONMENT": "production",
            },
            schedule_interval=dt(hours=1),
            kill_timeout=dt(hours=24),
            retry_interval=dt(hours=1),
            crit_time=dt(hours=24),
            aggr_hosts=STABLE_ONLY,
        ),
    ]:
        spine_helpers.create_run_universal_bundle_schedulers(sandbox, task)
