#!/usr/bin/perl
#
# Provides: freespace

use strict;
use File::stat;

# Ignoring sigpipe because of strange workaround in 'grep'
$SIG{PIPE} = 'IGNORE';

# $type = 0 - check in percents, 1 - in Mb
my $type = 0;
my $all_crit = '90%';
my $all_warn = '85%';
my $inodes_crit = '90%';
my $inodes_warn = '85%';

my ($all_crit_typed, $all_crit_type) = type_check($all_crit);
my ($all_warn_typed, $all_warn_type) = type_check($all_warn);
my ($inodes_crit_typed, $inodes_crit_type) = type_check($inodes_crit);
my ($inodes_warn_typed, $inodes_warn_type) = type_check($inodes_warn);

my $ignore_coredumps = 0;

# Checking if /home/yabs/server/data exists
my $check_bs = 0;
$check_bs = 1 if (system("mountpoint -q /home/yabs/server/data") == 0 and `hostname` !~ /^bsgm[0-9][0-9][a-z]/);

# Checking space available for next 2 biggest bases (exit if critical occured)
if ($check_bs == 1) {
    my $data_size = 0;
    my $data_avail = 0;
    if (`df -l -t hugetlbfs '/home/yabs/server/data' 2>/dev/null`) {
        my @hugepages = `cat /proc/meminfo | grep -i 'hugepage'`;
        my $t1 = 0;
        my $t2 = 0;
        foreach my $el (@hugepages) {
            if ($el =~ m/HugePages_Total/) {
                my @tmp = split(/\s+/, $el);
                $t1 = $tmp[1];
            }
            if ($el =~ m/Hugepagesize/) {
                my @tmp = split(/\s+/, $el);
                $t2 = $tmp[1];
            }
        }
        $data_avail = $t1 * $t2;
    }
    else {
        $data_avail = `df -lk /home/yabs/server/data 2> /dev/null | awk '(\$2 ~ /^[0-9]*\$/){print \$2; exit}'`;
    }
    $data_size = `du -skc /home/yabs/server/data/*.yabs 2> /dev/null | grep total | awk '{print \$1}'`;
    chomp $data_size;
    chomp $data_avail;
    my $data = $data_avail - $data_size;
    $data *= 1024;
    space_check("/home/yabs/server/data/*yabs", $data);
}

# Checking if some tmpfses mounted
my $check_tmpfs = 0;
my @tmpfses = ();
# Refactoring needed here:
#$check_tmpfs = 1 if (@tmpfses = `mount | awk '{print \$1 " " \$3;}' | grep tmpfs | grep -v '/home/yabs/server/data'`);

# Checking space available for next 2 biggest files in each tmpfs (exit if critical occured)
if ($check_tmpfs == 1) {
    foreach my $tmpfs (@tmpfses) {
        my $data_size = 0;
        my $data_avail = 0;
        my @mp = split(/\s/, $tmpfs);
        $data_avail = `df -lk 2> /dev/null | grep '$mp[1]' | awk '{print \$2}'`;
        $data_size = `du -sk $mp[1] | awk '{print \$1}'`;
        chomp $data_size;
        chomp $data_avail;
        my $data = $data_avail - $data_size;
        $data *= 1024;
        space_check($mp[1]."/*", $data);
    }
}

# All OK with /home/yabs/server/data and tmpfses, checking space for other partitions
my $inodes = 0;
my %criticals = ();
my %warnings = ();
my @mar = `df -l -x tmpfs -x hugetlbfs -x fuse.ceph-fuse -k 2> /dev/null; df -l /mnt/tmpfslogs -k 2> /dev/null`;
space_rotate($inodes, @mar);
$inodes = 1;
@mar = `df -l -x tmpfs -x hugetlbfs -x fuse.ceph-fuse -i 2> /dev/null; df -l /mnt/tmpfslogs -i 2> /dev/null`;
space_rotate($inodes, @mar);

# Prepairing output for Nagios
my $mcheck = 0;
my $message = 'OK';
my $status = 0;

foreach my $key (sort keys %warnings) {
    if ($key ne "/coredumps" || !$ignore_coredumps) {
        $status = 1;
        if ($mcheck == 0) {
            $message = "No space for $key: $warnings{$key} used!";
            $mcheck = 1;
        }
        else {
            $message = "More than one warning with no space!";
        }
    }
}

$mcheck = 0;

foreach my $key (sort keys %criticals) {
    if ($key ne "/coredumps" || !$ignore_coredumps) {
        $status = 2;
        if ($mcheck == 0) {
            $message = "No space for $key: $criticals{$key} used!";
            $mcheck = 1;
        }
        else {
            $message = "More than one critical with no space!";
        }
    }
}

mexit($status, $message);

sub space_rotate ($@);

sub space_rotate ($@) {
    my $inodes = $_[0];
    my @mar = @_;

    my %ever_checked;

    foreach my $el (@mar) {
        my $checked = 0;
        my @smar = split(/\s+/, $el);
        $smar[4] =~ s/%//;
        if ($smar[4] =~ m/^\d+/) {
            if ($inodes == 0) {
                $checked = space_compare($smar[5], $all_warn_type, 'all', 'warn', $smar[4], $smar[3], $all_warn_typed);
                $checked = space_compare($smar[5], $all_crit_type, 'all', 'crit', $smar[4], $smar[3], $all_crit_typed);
            }
            elsif ($inodes == 1) {
                $checked = space_compare($smar[5], $inodes_warn_type, 'inodes', 'warn', $smar[4], $smar[3], $inodes_warn_typed);
                $checked = space_compare($smar[5], $inodes_crit_type, 'inodes', 'crit', $smar[4], $smar[3], $inodes_crit_typed);
            }
            else {
                mexit(2,"Impossible error!");
            }
        }
    }
}

sub space_compare {
    my $name = $_[0];
    my $type = $_[1];
    my $param = $_[2];
    my $event_type = $_[3];
    my $used_pers = $_[4];
    my $avail_kb = $_[5];
    my $crit = $_[6]; # in Mb
    my $checked = 0;
    #  print $name." ".$type." ".$param." ".$event_type." ".$used_pers." ".$crit."\n";

    if (($name =~ m/$param/) || ($param =~ m/all/) || ($param =~ m/inodes/)) {
        $checked = 1;
        if ($type == 0) {
            if (($event_type eq 'warn') and ($used_pers >= $crit)) {
                $checked = 2;
                if ($param =~ m/inodes/) {
                    $warnings{$name} = $used_pers."% inodes";
                }
                else {
                    $warnings{$name} = $used_pers."%";
                }
            }
            if (($event_type eq 'crit') and ($used_pers >= $crit)) {
                $checked = 3;
                if ($param =~ m/inodes/) {
                    $criticals{$name} = $used_pers."% inodes";
                }
                else {
                    $criticals{$name} = $used_pers."%";
                }
            }
        }
        elsif ($type == 1) {
            $crit *= 1024;
            if (($event_type eq 'warn') and ($avail_kb <= $crit)) {
                my $avail_mb = int($avail_kb / 1024);
                $checked = 2;
                $warnings{$name} = $avail_mb."Mb";
            }
            if (($event_type eq 'crit') and ($avail_kb <= $crit)) {
                my $avail_mb = int($avail_kb / 1024);
                $checked = 3;
                $criticals{$name} = $avail_mb."Mb";
            }
        }
        else {
            mexit(2,'Impossible error occured %-|');
        }
    }
    return $checked;
}

sub type_check {
    my $value = $_[0];
    my ($ret, $type) = ();
    if ($value =~ m/^(\d+)%$/) {
        $ret = $1;
        $type = 0;
    }
    elsif ($value =~ m/^(\d+)M$/) {
        $ret = $1;
        $type = 1;
    }
    else {
        mexit(2, "Error in config file!");
    }
    return ($ret, $type);
}

# Checking some tmps or hugetlbsfs with phantom bases
sub space_check {
    my $path = $_[0];
    my $data_size = $_[1];
    my $message = "No needed space left in tmpfs!";
    my $baseornot = 0;
    $message = "No needed space left for bases!" and $baseornot = 1 if ($path =~ m!/home/yabs/server/data!);
    my $alloc_check = 0;
    if ($baseornot) {
        if (`df -l -t hugetlbfs '/home/yabs/server/data' 2> /dev/null`) {
            my @hugepages = `cat /proc/meminfo | grep -i 'hugepage'`;
            my $t1 = 0;
            my $t2 = 0;
            foreach my $el (@hugepages) {
                if ($el =~ m/HugePages_Free/) {
                    my @tmp = split(/\s+/, $el);
                    $t1 = $tmp[1];
                }
                if ($el =~ m/Hugepagesize/) {
                    my @tmp = split(/\s+/, $el);
                    $t2 = $tmp[1];
                }
            }
            $alloc_check = $t1 * $t2;
        }
        else {
            $alloc_check = `df -lk '/home/yabs/server/data' 2> /dev/null | awk '(\$4 ~ /^[0-9]*\$/){print \$4; exit}'`;
        }
        $alloc_check *= 1024;
    }
    my @bases = split(/\n/, `ls $path 2> /dev/null`);
    my @sizes = ();
    my %sizess = ();
    foreach my $el (@bases) {
        chomp $el;
        my $stat = stat($el);
        if ($stat) {
            $sizess{$stat->size} = $el;
            push (@sizes, $stat->size);
        }
    }
    my $count = 1;
    my $space_needed = 0;
    my $spare_base_count_needed = 2;
    foreach my $key (sort {$b <=> $a} keys %sizess) {
        next if ($count > $spare_base_count_needed);
        $space_needed += $key;
        $count++;
    }
    if ($space_needed >= $data_size) {
        mexit(2, "@{[human_size($data_size)]} <= @{[human_size($space_needed)]} in $path; $message");
    }
    elsif (($alloc_check != 0) && ($space_needed >= $alloc_check)) {
        mexit(1, "@{[human_size($data_size)]} in $path; Deleted base found!");
    }
}

# Exit correctly
sub mexit {
    my ($status, $msg) = @_;
    $msg = substr($msg, 0, 1024);
    my $tmpfile="/home/monitor/agents/tmp/freespace_flapdetect";
    if ($status == 2 and flap_detect($tmpfile)) {
        print "PASSIVE-CHECK:freespace;$status;$msg\n";
    }
    else {
        $status=1 if $status;
        print "PASSIVE-CHECK:freespace;$status;$msg\n";
    }
    # if status is ok and flapfile truncate it
    if ($status == 0 and -f $tmpfile) {
        open(FF,">",$tmpfile);
        print FF "0";
        close(FF);
        chmod(0666,$tmpfile);
    }
    exit 0;
}

sub flap_detect {
    my ($tmpfile) = @_;
    if ( -f $tmpfile ) {
        open(FF,"<",$tmpfile) or return 0;
        my $flap_count=<FF>;
        close(FF);
        chomp($flap_count);
        if ($flap_count > 10) {
            return 1;
        }
        else {
            $flap_count++;
            open(FF,">",$tmpfile);
            print FF $flap_count;
            close(FF);
            chmod(0666,$tmpfile);
            return 0;
        }
    }
    else {
        open(FF,">",$tmpfile);
        print FF "1";
        close(FF);
        chmod(0666,$tmpfile);
        return 0;
    }
}

sub human_size {
    my $size = shift;
    my $suffix = "B";
    if ($size > 1024) {
        $size /= 1024;
        $suffix = "KB";
    }
    if ($size > 1024) {
        $size /= 1024;
        $suffix = "MB";
    }
    if ($size > 1024) {
        $size /= 1024;
        $suffix = "GB";
    }
    return sprintf("%.1f", $size) . $suffix;
}
