#!/usr/bin/env python

import argparse
import json
import logging
import logging.config
import os
import socket
import subprocess
import sys

logger = logging.getLogger(__name__)


def get_conf_name(conf_path):
    return os.path.splitext(os.path.basename(conf_path))[0]


def call_juggler_queue_event(conf_path, retcode, description, dry_run):
    cmd = [
        'sudo', 'juggler_queue_event',
        '--host={}'.format(socket.getfqdn()),
        '-n', 'push_{}'.format(get_conf_name(conf_path)),
        '-s', str(retcode),
        '-d', description
    ]

    logger.debug('Call to juggler: {}'.format(' '.join(cmd)))

    if not dry_run:
        subprocess.call(cmd)
    else:
        logger.debug('Dry run.')
        print ' '.join(cmd)


class PushClientChecker:
    def __init__(self, send_time, commit_time, lag_size, dry_run=False):
        self.send_time = send_time
        self.commit_time = commit_time
        self.lag_size = lag_size
        self.dry_run = dry_run

    def check_config_dir(self, conf_dir):
        logger.info('Looking for push-client configs in directory: {}'.format(conf_dir))

        if not os.path.exists(conf_dir):
            logger.error('Config directory does not exist')
            return
        if not os.path.isdir(conf_dir):
            logger.error('Path to config directory is not a directory.')
            return

        configs = os.listdir(conf_dir)

        for config in configs:
            logger.info('Checking status for {}'.format(config))
            conf_path = os.path.join(conf_dir, config)
            retcode, description = self.check_config(conf_path)
            call_juggler_queue_event(conf_path, retcode, description, self.dry_run)
            logger.info('Done.')

        logger.info('All configs were checked successfully')

    def check_config(self, conf_path):
        json_str, retcode = self.call_push_client_status(conf_path)

        description = 'Ok'
        if retcode:
            retcode = 2
            description = self.generate_error_msg(json_str)

        logger.info('Status: {}, description: {}'.format(retcode, description))
        if retcode:
            logger.warn('Push-client returned error.')
            logger.warn('JSON:\n{}'.format(json_str))

        return retcode, description

    def call_push_client_status(self, conf_path):
        cmd = [
            'push-client',
            '-c', conf_path,
            '--status', '--json',
            '--check:send-time={}'.format(self.send_time),
            '--check:commit-time={}'.format(self.commit_time),
            '--check:lag-size={}'.format(self.lag_size),
        ]

        logger.debug('Call to push-client: {}'.format(' '.join(cmd)))

        proc = subprocess.Popen(cmd, stdout=subprocess.PIPE, stderr=subprocess.PIPE)
        out, err = proc.communicate()
        return out, proc.returncode

    def generate_error_msg(self, json_str):
        try:
            data = json.loads(json_str)
            if not data:
                return 'empty JSON (check if push-client service is running)'

            errors = []
            for log_data in data:
                errors += self.generate_log_err_msg(log_data)

            if not errors:
                return 'unknown error'
            return ', '.join(errors)

        except:
            exc_type, exc_value, exc_traceback = sys.exc_info()
            return 'Exception: [{}] {}'.format(exc_type.__name__, exc_value)

    def generate_log_err_msg(self, log_data):
        if log_data['status'] == 1:
            return []

        if log_data['type'] != 'file':
            return ['[{}] wrong status type: {}'.format(log_data['name'], log_data['type'])]

        if 'errors' in log_data:
            reasons = log_data['errors']
        else:
            reasons = []
            if 0 < self.commit_time < log_data['commit delay']:
                reasons.append('commit delay above threshold ({} > {})'.format(
                    log_data['commit delay'], self.commit_time)
                )

            if 0 < self.lag_size < log_data['lag']:
                reasons.append('lag is above threshold ({} > {})'.format(log_data['lag'], self.lag_size))

            send_delay = log_data['last commit time'] + log_data['commit delay'] - log_data['last send time']
            if 0 < self.send_time < send_delay:
                reasons.append('send delay above threshold ({} > {})'.format(send_delay, self.send_time))

            if log_data['file offset'] > log_data['file size']:
                reasons.append('file has been truncated')

        if not reasons:
            reasons = ['unknown reason']

        return ['[{}] {}'.format(log_data['name'], ', '.join(reasons))]


def parse_args():
    parser = argparse.ArgumentParser(description='Monitor push-client')
    parser.add_argument('--config_dir', default=None,
                        help='Directory with push client configs')
    parser.add_argument('--send_time', type=int, default=10*60,
                        help='Threshold for delay between send attempts (seconds)')
    parser.add_argument('--commit_time', type=int, default=30*60*60,
                        help='Threshold for delay between commits (seconds)')
    parser.add_argument('--lag_size', type=int, default=-1,
                        help='Threshold for lag size (bytes)')
    parser.add_argument('--dry_run', action='store_true',
                        help='Don\'t post status to juggler, just print command instead.')
    parser.add_argument('--log-config', help='Logger config file', required=True)

    args = parser.parse_args()
    if not args.config_dir:
        with open("/etc/default/push-client") as push_client_config:
            for line in push_client_config:
                name, value = line.strip().split('=', 1)
                if name == "DAEMON_CONF":
                    args.config_dir = os.path.dirname(value.strip('"'))

        if not args.config_dir:
            raise Exception("No config dir in parameters or push-client config")

    return args


def main():
    args = parse_args()
    logging.config.fileConfig(args.log_config, disable_existing_loggers=False)

    try:
        checker = PushClientChecker(args.send_time, args.commit_time, args.lag_size, args.dry_run)
        checker.check_config_dir(args.config_dir)
    except:
        logger.exception('Failed with exception.')


if __name__ == '__main__':
    main()

