import logging

from crypta.lib.python import templater
from crypta.lib.python.juggler.juggler_helpers import report_event_to_juggler
from crypta.prism.lib.config import (
    config,
    environment,
)


logger = logging.getLogger(__name__)

correlation_query = '''
{% set first_id = first_id or 'first' %}
{% set second_id = second_id or 'second' %}
{% set key = key or '`key`' %}
INSERT INTO `{{ output_table }}` WITH TRUNCATE
SELECT
    COUNT(*) AS matched_count,
    CORRELATION(first.`value`, second.`value`) AS `correlation`,
FROM ${{ first_id }}_prepared AS first
INNER JOIN ${{ second_id }}_prepared AS second
USING({{ key }});
'''

correlation_metric_lower_bound_query = '''
INSERT INTO `{{ output_table }}` WITH TRUNCATE
SELECT
    `source`,
    AVG(`correlation`) -
        3.0 * STDDEV(`correlation`) AS correlation_lower_bound,
    AVG(matched_count) -
        3.0 * STDDEV(matched_count) AS matched_count_lower_bound,
FROM (
    SELECT
        `source`,
        `correlation`,
        matched_count,
        RANK() OVER w AS days_ago,
    FROM `{{ metrics_table }}`
    WINDOW w AS (
        PARTITION BY `source` ORDER BY fielddate DESC
    )
)
WHERE days_ago > 0
    AND days_ago <= {{ days_for_metric_comparison }}
GROUP BY `source`;
'''

prior_utils_query = '''
$get_screen_info_inner = ($szm_screen, $szm_ratio) -> {
    $dimentions = String::SplitToList($szm_screen, 'x');
    $x = CAST($dimentions[0] AS Uint32);
    $y = CAST($dimentions[1] AS Uint32);
    $height = CAST(MIN_OF($x, $y) AS String);
    $width = CAST(MAX_OF($x, $y) AS String);
    RETURN $height || 'x' || $width || ';' || CAST($szm_ratio AS String);
};

$get_screen_info = ($szm_screen, $szm_ratio) -> {
    RETURN IF(
        $szm_screen IS NOT NULL AND $szm_ratio IS NOT NULL,
        $get_screen_info_inner($szm_screen, $szm_ratio),
        NULL,
    );
};

$extend = ($field) -> {
    RETURN IF (
        $field IS NOT NULL,
        [$field, '*'],
        ['*'],
    );
};
'''


def report_correlation_metrics(
    yt_client,
    yql_client,
    transaction,
    metrics,
    metrics_table,
    service_name,
    host=config.CRYPTA_ML_JUGGLER_HOST,
):
    with yt_client.TempTable() as metric_lower_bound:
        yql_client.execute(
            templater.render_template(
                correlation_metric_lower_bound_query,
                vars={
                    'output_table': metric_lower_bound,
                    'days_for_metric_comparison': config.DAYS_FOR_QUALITY_METRICS_COMPARISON,
                    'metrics_table': metrics_table,
                },
            ),
            title='YQL Calculate lower bound',
            transaction=str(transaction.transaction_id),
        )

        bad_metrics_messages = []

        for lower_bound in yt_client.read_table(metric_lower_bound):
            metrics_for_source = list(filter(lambda metric: metric['source'] == lower_bound['source'], metrics))[0]

            for metric_type in ['matched_count', 'correlation']:
                message = 'For table {} metric {} is {} (lower bound is {})'.format(
                    lower_bound['source'],
                    metric_type,
                    metrics_for_source[metric_type],
                    lower_bound['{}_lower_bound'.format(metric_type)],
                )
                logger.info(message)

                if metrics_for_source[metric_type] < lower_bound['{}_lower_bound'.format(metric_type)]:
                    bad_metrics_messages.append(message)

        if environment.ENVIRONMENT == 'production':
            if len(bad_metrics_messages) == 0:
                report_event_to_juggler(
                    status='OK',
                    service=service_name,
                    host=host,
                    logger=logger,
                )
            else:
                report_event_to_juggler(
                    status='WARN',
                    service=service_name,
                    host=host,
                    description='. '.join(bad_metrics_messages),
                    logger=logger,
                )
