#!/usr/bin/env python
# -*- coding: utf-8 -*-

import functools
import logging

from crypta.lib.python.logging import logging_helpers
from crypta.prism.lib.nirvana import utils
from crypta.prism.services.offline_weighting.lib import (
    cluster_stats,
    clusters,
    lookalike_samples,
    performance_metrics,
    quality_metrics,
    user_weights,
    user_weights_formatting,
)


import nirvana.job_context as nv

logger = logging.getLogger(__name__)


def main():
    logging_helpers.configure_stderr_logger(logging.getLogger(), level=logging.INFO)
    logger.info('Prism offline weighting')

    context = nv.context()

    parameters = context.get_parameters()
    custom_output_dir = parameters.get('custom-output-dir') if parameters.get('custom-output-dir') != '' else None
    date = parameters.get('date')
    yt_client = utils.get_yt_client(nv_params=parameters)
    yql_client = utils.get_yql_client(nv_params=parameters)

    outputs = context.get_outputs()
    output_table = None if not outputs.has('output_table') else outputs.get('output_table')

    tasks_dict = {
        'calculate_cluster_stats': functools.partial(
            cluster_stats.calculate_by_date,
            yt_client=yt_client,
            yql_client=yql_client,
            date=date,
            nv_output_table_file=output_table,
            custom_output_dir=custom_output_dir,
        ),
        'calculate_clusters': functools.partial(
            clusters.calculate_by_date,
            yt_client=yt_client,
            yql_client=yql_client,
            date=date,
            custom_output_dir=custom_output_dir,
        ),
        'calculate_user_weights': functools.partial(
            user_weights.calculate_by_date,
            yt_client=yt_client,
            yql_client=yql_client,
            date=date,
            nv_output_table_file=output_table,
            custom_output_dir=custom_output_dir,
        ),
        'format_user_weights_to_bigb': functools.partial(
            user_weights_formatting.to_bigb_by_date,
            yt_client=yt_client,
            date=date,
            custom_output_dir=custom_output_dir,
        ),
        'get_lookalike_samples': functools.partial(
            lookalike_samples.get_by_date,
            yt_client=yt_client,
            yql_client=yql_client,
            date=date,
            custom_output_dir=custom_output_dir,
        ),
        'prepare_adv_table': functools.partial(
            quality_metrics.prepare_adv_table,
            yt_client=yt_client,
            yql_client=yql_client,
            date=date,
        ),
        'prepare_gmv_table': functools.partial(
            quality_metrics.prepare_gmv_table,
            yt_client=yt_client,
            yql_client=yql_client,
            date=date,
        ),
        'prepare_yandex_google_visits_table': functools.partial(
            quality_metrics.prepare_yandex_google_visits_table,
            yt_client=yt_client,
            yql_client=yql_client,
            date=date,
        ),
        'send_auc_metrics': functools.partial(
            quality_metrics.send_auc_metrics,
            yt_client=yt_client,
            yql_client=yql_client,
            date=date,
            custom_output_dir=custom_output_dir,
        ),
        'send_correlation_metrics': functools.partial(
            quality_metrics.send_correlation_metrics,
            yt_client=yt_client,
            yql_client=yql_client,
            date=date,
        ),
        'update_performance_metrics': functools.partial(
            performance_metrics.update,
            yt_client=yt_client,
            yql_client=yql_client,
            date=date,
        ),
    }

    job_name = parameters.get('job_name')
    logger.info('Job name: {}'.format(job_name))

    if job_name in tasks_dict:
        tasks_dict[job_name]()
    else:
        logger.warn('Unknown job_name "{}"'.format(job_name))
        exit(1)


if __name__ == '__main__':
    main()
