from datetime import timedelta

from crypta.lib.python import templater
from crypta.lib.python.nirvana.nirvana_helpers.nirvana_transaction import NirvanaTransaction
from crypta.lib.python.yt import yt_helpers
from crypta.prism.lib.config import config
from crypta.prism.services.offline_weighting.lib import table_paths
from crypta.profile.lib import date_helpers

lookalike_samples_query = '''
PRAGMA AnsiInForEmptyOrNullableItemsCollections;

$tariff_stat = (
    SELECT
        passport_uid AS puid,
        order_cnt,
        Yson::ConvertToUint64Dict(tariff_class_stat) AS tariff_stats,
    FROM `{{ taxi_user_profile_table }}`
);

$tariff_stat = (
    SELECT
        puid,
        order_cnt,
        tariff_stat.0 AS tariff,
        tariff_stat.1 AS tariff_cnt,
    FROM $tariff_stat
    FLATTEN DICT BY tariff_stats AS tariff_stat
    WHERE tariff_stat.0 in {{ ultima_tariffs }}
);

$by_puid = (
    SELECT
        puid,
        SOME(order_cnt) AS order_cnt,
        CAST(SUM(tariff_cnt) AS Double) / SOME(order_cnt) AS business_order_ratio,
    FROM $tariff_stat
    GROUP BY puid
);

$ultima_taxi_users = (
    SELECT CAST(indevice_yandexuid.yandexuid AS String) AS yandexuid
    FROM $by_puid AS taxi
    INNER JOIN `{{ indevice_yandexuid_matching_table }}` AS indevice_yandexuid
    ON taxi.puid == indevice_yandexuid.id
    WHERE indevice_yandexuid.id_type == 'puid'
    GROUP BY indevice_yandexuid.yandexuid
    HAVING MAX(taxi.order_cnt) >= {{ min_order_cnt }}
        AND MAX_BY(taxi.business_order_ratio, taxi.order_cnt) > {{ min_business_order_ratio }}
);

$lavka_users = (
    SELECT yandexuid_cryptaid.id AS yandexuid
    FROM `{{ tx_lavka_table }}` AS tx
    INNER JOIN `{{ yandexuid_cryptaid_matching_table }}` AS yandexuid_cryptaid
    ON tx.CryptaId == yandexuid_cryptaid.target_id
    WHERE tx.Status == 'Finished'
        AND String::SplitToList(CAST(DateTime::FromSeconds(
            tx.`Timestamp`) AS String), 'T')[0] >= '{{ three_months_ago }}'
    GROUP BY yandexuid_cryptaid.id
);

$ultima_taxi_and_lavka_users = (
    SELECT ultima.yandexuid AS yandexuid
    FROM $ultima_taxi_users AS ultima
    INNER JOIN $lavka_users AS lavka
    USING(yandexuid)
);

$elite_cards = (
    SELECT CAST(yandexuid AS String) AS yandexuid
    FROM `{{ profiles_table }}`
    WHERE ListHas(Yson::ConvertToUint64List(audience_segments), {{ mastercard_we_audience_segment_id }})
        OR ListHas(Yson::ConvertToUint64List(audience_segments), {{ visa_infinite_audience_segment_id }})
);

INSERT INTO `{{ lookalike_samples_table }}` WITH TRUNCATE
SELECT
    'elite_cards' AS GroupID,
    yandexuid AS IdValue,
    'yandexuid' AS IdType,
FROM $elite_cards
UNION ALL
SELECT
    'ultima_taxi_and_lavka_users' AS GroupID,
    yandexuid AS IdValue,
    'yandexuid' AS IdType,
FROM $ultima_taxi_and_lavka_users;
'''


def get(
    yql_client,
    date,
    lookalike_samples_table,
    transaction,
):
    yql_client.execute(
        templater.render_template(
            lookalike_samples_query,
            vars={
                'mastercard_we_audience_segment_id': config.MASTERCARD_WE_AUDIENCE_SEGMENT_ID,
                'visa_infinite_audience_segment_id': config.VISA_INFINITE_AUDIENCE_SEGMENT_ID,
                'ultima_tariffs': config.LAL_SAMPLE_TAXI_ULTIMA_TARIFFS,
                'min_business_order_ratio': config.LAL_SAMPLE_TAXI_MIN_BUSINESS_ORDER_RATIO,
                'min_order_cnt': config.LAL_SAMPLE_TAXI_MIN_ORDER_CNT,
                'three_months_ago': date_helpers.get_date_from_past(current_date=date, months=3),
                'taxi_user_profile_table': config.TAXI_USER_PROFILE_TABLE,
                'tx_lavka_table': config.TX_LAVKA_TABLE,
                'profiles_table': config.YANDEXUID_EXPORT_PROFILES_14_DAYS_TABLE,
                'yandexuid_cryptaid_matching_table': config.YANDEXUID_CRYPTAID_MATCHING_TABLE,
                'indevice_yandexuid_matching_table': config.INDEVICE_YANDEXUID_MATCHING_TABLE,
                'lookalike_samples_table': lookalike_samples_table,
            },
        ),
        title='YQL Prism get lookalike samples',
        transaction=str(transaction.transaction_id),
    )


def get_by_date(yt_client, yql_client, date, custom_output_dir=None):
    with NirvanaTransaction(yt_client) as transaction:
        tables = table_paths.resolve(custom_output_dir, date)

        get(
            yql_client=yql_client,
            date=date,
            lookalike_samples_table=tables['lookalike_samples'],
            transaction=transaction,
        )

        if custom_output_dir is None:
            yt_helpers.set_ttl(tables['lookalike_samples'], timedelta(days=config.PRISM_LOOKALIKE_SAMPLE_TTL_DAYS), yt_client=yt_client)
