import logging
import os

from crypta.lib.python import (
    prism_quality,
    templater,
)
from crypta.lib.python.nirvana.nirvana_helpers.nirvana_transaction import NirvanaTransaction
from crypta.lib.python.yt import yt_helpers
from crypta.prism.lib.config import config
from crypta.prism.lib.utils import metrics as metric_utils
from crypta.prism.services.offline_weighting.lib import table_paths
from crypta.profile.lib import date_helpers


logger = logging.getLogger(__name__)

lookalike_preparation_query = '''
${{ id }}_prepared = (
    SELECT
        Yandexuid AS key,
        RANK() OVER w AS value,
    FROM `{{ input_table }}`
    WHERE GroupID == '{{ group_id }}'
    WINDOW w AS (
        ORDER BY Score
    )
);
'''

income_preparation_query = '''
${{ id }}_prepared = (
    SELECT
        yandexuid AS `key`,
        RANK() OVER w AS `value`,
    FROM `{{ input_table }}`
    WINDOW w AS (
        ORDER BY Coalesce(Yson::ConvertToDoubleDict(income_5_segments)['C2'], 0)
    )
);
'''

prism_preparation_query = '''
${{ id }}_prepared = (
    SELECT
        yandexuid AS `key`,
        RANK() OVER w AS `value`,
    FROM `{{ input_table }}`
    WINDOW w AS (
        ORDER BY CAST(cluster AS Double) + rank_in_cluster
    )
);
'''

os_split_template = """
CASE
    WHEN {os_column} == 'Windows' THEN 'windows'
    WHEN {os_column} == 'Android' THEN 'android'
    WHEN {os_column} == 'iOS' THEN 'ios'
    WHEN {os_column} == 'MacOS' THEN 'macos'
    ELSE 'other'
END AS os
"""

chevent_cost_query = """
PRAGMA AnsiInForEmptyOrNullableItemsCollections;

$distinct = (
    SELECT DISTINCT
        uniqid AS yandexuid,
        logid,
        eventcost AS cost,
        {os_split_template}
    FROM `{chevent_log_table}`
    WHERE placeid in (542, 1542)
        AND fraudbits == 0
);

INSERT INTO `{output_table}`
WITH TRUNCATE

SELECT
    yandexuid,
    os,
    SUM(cost) AS cost,
FROM $distinct
GROUP BY yandexuid, os
ORDER BY yandexuid;
"""

gmv_cost_query = """
INSERT INTO `{output_table}`
WITH TRUNCATE

SELECT
    CAST(ecom.yandexuid AS Uint64) AS yandexuid,
    {os_split_template},
    SUM(NANVL(ecom.price / ecom.multiplier * takerate.takerate, 0.0)) AS cost,
FROM `{sales_no_external_table}` AS ecom
INNER JOIN `{takerate_table}` AS takerate
ON ecom.hid == takerate.hyper_id
GROUP BY ecom.yandexuid, ecom.os_family AS os_family
ORDER BY yandexuid;
"""

yandex_google_visits_query = """
PRAGMA AnsiInForEmptyOrNullableItemsCollections;

$Y_SEARCH_RESULT = [2, 13, 181];
$G_SEARCH_RESULT = [3, 68, 224];

$visits_uids = (
    SELECT
        CAST(user_id AS Uint64) AS yandexuid,
        IF(search_engine_id IN $Y_SEARCH_RESULT, "yandex", "google") AS system,
        {os_split_template}
    FROM `{traff_table}`
    WHERE (search_engine_id IN $Y_SEARCH_RESULT OR search_engine_id IN $G_SEARCH_RESULT)
        AND Geo::RoundRegionById(CAST(geo_id AS Int32), 'country').short_en_name == "RU"
);

INSERT INTO `{output_table}`
WITH TRUNCATE

SELECT
    yandexuid,
    os,
    COUNT_IF(system == 'yandex') AS yandex_visits,
    COUNT_IF(system == 'google') AS google_visits,
FROM $visits_uids
GROUP BY yandexuid, os
ORDER BY yandexuid;
"""


def send_correlation_metrics(yt_client, yql_client, date):
    with NirvanaTransaction(yt_client) as transaction, \
            yt_client.TempTable() as income_correlation_table, \
            yt_client.TempTable() as prism_weights_correlation_table, \
            yt_client.TempTable() as ultima_taxi_and_lavka_users_correlation_table, \
            yt_client.TempTable() as elite_cards_correlation_table:

        previous_date = date_helpers.get_date_from_past(current_date=date, days=1)

        sources = {
            'lookalike': [
                {
                    'first': os.path.join(config.PRISM_LAL_DIR, previous_date),
                    'second': os.path.join(config.PRISM_LAL_DIR, date),
                    'group_id': 'elite_cards',
                    'output_table': elite_cards_correlation_table,
                },
                {
                    'first': os.path.join(config.PRISM_LAL_DIR, previous_date),
                    'second': os.path.join(config.PRISM_LAL_DIR, date),
                    'group_id': 'ultima_taxi_and_lavka_users',
                    'output_table': ultima_taxi_and_lavka_users_correlation_table,
                },
            ],
            'prism': [
                {
                    'first': os.path.join(config.PRISM_OFFLINE_USER_WEIGHTS_DIR, previous_date),
                    'second': os.path.join(config.PRISM_OFFLINE_USER_WEIGHTS_DIR, date),
                    'output_table': prism_weights_correlation_table,
                },
            ],
            'income': [
                {
                    'first': os.path.join(config.YANDEXUID_PROFILES_EXPORT_DIR, previous_date),
                    'second': os.path.join(config.YANDEXUID_PROFILES_EXPORT_DIR, date),
                    'output_table': income_correlation_table,
                },
            ],
        }

        table_type_to_query_template = {
            'lookalike': lookalike_preparation_query,
            'prism': prism_preparation_query,
            'income': income_preparation_query,
        }

        query_parts = []

        for source_type in sources:
            for table_pairs in sources[source_type]:
                for table_number in ['first', 'second']:
                    group_id = table_pairs.get('group_id') or source_type
                    query_parts.append(
                        templater.render_template(
                            table_type_to_query_template[source_type],
                            vars={
                                'input_table': table_pairs[table_number],
                                'id': '{}_{}'.format(table_number, group_id),
                                'group_id': group_id,
                            },
                        ),
                    )

                query_parts.append(
                    templater.render_template(
                        metric_utils.correlation_query,
                        vars={
                            'output_table': table_pairs['output_table'],
                            'first_id': 'first_{}'.format(group_id),
                            'second_id': 'second_{}'.format(group_id),
                        },
                    ),
                )

        yql_client.execute(
            '\n'.join(query_parts),
            title='YQL Calculate correlation',
            transaction=str(transaction.transaction_id),
        )

        metrics = []

        for source_type in sources:
            for table_pairs in sources[source_type]:
                metrics_for_source = next(yt_client.read_table(table_pairs['output_table']))
                metrics_for_source.update({
                    'source': table_pairs.get('group_id') or source_type,
                })
                metrics.append(metrics_for_source)

        yt_helpers.write_stats_to_yt(
            yt_client=yt_client,
            table_path=config.PRISM_DAY_TO_DAY_CORRELATION_TABLE,
            data_to_write=metrics,
            schema={
                'source': 'string',
                'matched_count': 'uint64',
                'correlation': 'double',
            },
            date=date,
            fielddate='fielddate',
        )

        metric_utils.report_correlation_metrics(
            yt_client=yt_client,
            yql_client=yql_client,
            transaction=transaction,
            metrics=metrics,
            metrics_table=config.PRISM_DAY_TO_DAY_CORRELATION_TABLE,
            service_name='prism_correlation',
        )


def prepare_gmv_table(yt_client, yql_client, date):
    with NirvanaTransaction(yt_client) as transaction:
        yql_client.execute(
            query=gmv_cost_query.format(
                os_split_template=os_split_template.format(os_column='os_family'),
                sales_no_external_table=os.path.join(config.MARKET_SALES_NO_EXTERNAL_FOLDER, date),
                takerate_table=config.MARKET_TAKERATE_TABLE,
                output_table=os.path.join(config.PRISM_GMV_DIRECTORY, date),
            ),
            transaction=transaction.transaction_id,
            title='YQL Calculate prism quality metrics prerequisites',
        )


def prepare_adv_table(yt_client, yql_client, date):
    with NirvanaTransaction(yt_client) as transaction:
        yql_client.execute(
            query=chevent_cost_query.format(
                os_split_template=os_split_template.format(os_column='detaileddevicetype'),
                chevent_log_table=os.path.join(config.BS_CHEVENT_COOKED_LOG, date),
                output_table=os.path.join(config.PRISM_CHEVENT_LOG_DIRECTORY, date),
            ),
            transaction=transaction.transaction_id,
            title='YQL Calculate prism quality metrics prerequisites',
        )


def prepare_yandex_google_visits_table(yt_client, yql_client, date):
    with NirvanaTransaction(yt_client) as transaction:
        yql_client.execute(
            query=yandex_google_visits_query.format(
                os_split_template=os_split_template.format(os_column='os_family'),
                traff_table=os.path.join(config.TRAFFIC_V3_DIR, date),
                output_table=os.path.join(config.PRISM_YANDEX_GOOGLE_VISITS_DIRECTORY, date),
            ),
            transaction=transaction.transaction_id,
            title='YQL Calculate prism quality metrics prerequisites',
        )


def send_auc_metrics(yt_client, yql_client, date, custom_output_dir=None):
    with NirvanaTransaction(yt_client) as transaction:
        tables = table_paths.resolve(custom_output_dir, date)

        metrics = prism_quality.check_quality(
            yt_client=yt_client,
            yql_client=yql_client,
            tables_to_check=[tables['user_weights']],
            dates=[date],
            output_dirs=[config.PRISM_QUALITY_DIRECTORY],
            transaction=transaction,
        )[0]

        for row in metrics:
            row['metric'] = float(row['metric'])

        yt_helpers.write_stats_to_yt(
            yt_client=yt_client,
            table_path=config.DATALENS_PRISM_PRODUCTION_METRICS_TABLE,
            data_to_write=metrics,
            schema={
                'fielddate': 'string',
                'os': 'string',
                'metric_type': 'string',
                'metric': 'double',
            },
        )
