from datetime import timedelta
import json
import os
import time

from crypta.lib.python.nirvana.nirvana_helpers.nirvana_transaction import NirvanaTransaction
from crypta.lib.python.yt import yt_helpers
from crypta.prism.lib.config import config
from crypta.prism.services.offline_weighting.lib import table_paths


def get_output_row(
    identifier_key_bigb,
    identifier_value,
    weight_and_cluster,
    cluster,
    prism_segment,
    timestamp,
):
    assert identifier_key_bigb in ['crypta_id', 'yandex_uid'], 'Unsupported identifier'
    assert cluster is not None, 'Nullable cluster'

    return {
        'value': json.dumps(
            {
                identifier_key_bigb: identifier_value,
                'timestamp': int(timestamp),
                'commands': [
                    {
                        'add': {
                            'items': [
                                {
                                    'keyword_id': config.BIGB_PRISM_KEYWORD,
                                    'update_time': int(timestamp),
                                    'uint_values': [
                                        weight_and_cluster,
                                        cluster,
                                        prism_segment,
                                    ],
                                },
                            ],
                        },
                    },
                ],
            },
        ),
    }


class UserWeightToBigbMapper(object):
    prism_segment_to_id = {'p1': 1, 'p2': 2, 'p3': 3, 'p4': 4, 'p5': 5}

    def __init__(self):
        self.timestamp = time.time()

    def get_prism_segment_id(self, segment):
        assert segment, 'Expect valid prism segment value: {}'.format(segment)

        if segment in self.prism_segment_to_id:
            return self.prism_segment_to_id[segment]
        else:
            return config.BIGB_UNKNOWN_CLUSTER_VALUE

    def __call__(self, row):
        cluster = int(row['cluster']) if row['cluster'] != 'Unknown' else config.BIGB_UNKNOWN_CLUSTER_VALUE
        cluster_prefix = cluster * config.BIGB_CLUSTER_MULT

        uint_weight = int(dict(row['longterm_metrics'])['segment_week_m1'] * config.BIGB_FLOAT_TO_UINT_MULT)
        weight_and_cluster = cluster_prefix + uint_weight

        for identifier_key_bigb, identifier_key in [
            ['crypta_id', 'crypta_id'],
            ['yandex_uid', 'yandexuid'],
        ]:
            if row[identifier_key] is not None:
                yield {
                    'identifier_key_bigb': identifier_key_bigb,
                    'identifier_value': int(row[identifier_key]),
                    'weight_and_cluster': weight_and_cluster,
                    'cluster': cluster,
                    'prism_segment': self.get_prism_segment_id(row['prism_segment']),
                    'timestamp': self.timestamp,
                }


class DropDuplicatesReducer(object):
    def __call__(self, key, rows):
        output_row = next(iter(rows))
        yield get_output_row(
            identifier_key_bigb=output_row['identifier_key_bigb'],
            identifier_value=output_row['identifier_value'],
            weight_and_cluster=output_row['weight_and_cluster'],
            cluster=output_row['cluster'],
            prism_segment=output_row['prism_segment'],
            timestamp=output_row['timestamp'],
        )


def to_bigb(
    yt_client,
    date,
    user_weights_bigb_output,
    user_weights=None,
):
    user_weights = user_weights or os.path.join(config.PRISM_OFFLINE_USER_WEIGHTS_DIR, date)

    yt_helpers.create_empty_table(
        yt_client=yt_client,
        path=user_weights_bigb_output,
        schema=[
            {'name': 'value', 'type': 'string'},
        ],
    )

    yt_client.run_map_reduce(
        UserWeightToBigbMapper(),
        DropDuplicatesReducer(),
        user_weights,
        user_weights_bigb_output,
        reduce_by=('identifier_key_bigb', 'identifier_value'),
    )


def to_bigb_by_date(yt_client, date, custom_output_dir=None):
    with NirvanaTransaction(yt_client):
        tables = table_paths.resolve(custom_output_dir, date)

        to_bigb(
            yt_client=yt_client,
            date=date,
            user_weights_bigb_output=tables['user_weights_bigb'],
            user_weights=tables['user_weights'],
        )

        if custom_output_dir is None:
            yt_helpers.set_ttl(tables['user_weights_bigb'], timedelta(days=config.USER_WEIGHTS_BIGB_TTL_DAYS), yt_client=yt_client)
