from datetime import timedelta
import os

from crypta.lib.python import templater
from crypta.lib.python.nirvana.nirvana_helpers.nirvana_transaction import NirvanaTransaction
from crypta.lib.python.yt import yt_helpers
from crypta.prism.lib.config import config
from crypta.prism.lib.utils import metrics


features_and_cluster_by_user_query = '''
$sessions_clean_columns = '{{ user_sessions_clean }}';
$sessions_frauds_columns = '{{ user_sessions_frauds }}';
$sessions_staff_columns = '{{ user_sessions_staff }}';
$prism_path = '{{ prism_weights }}';

$yandexuid_icookie_matching = '{{ yandexuid_icookie_matching }}';

-- Clusters
$active_icookie = (
    SELECT
        -- Remove 'y' prefix
        DISTINCT SUBSTRING(key, 1) AS icookie,
    FROM CONCAT(
        $sessions_clean_columns VIEW raw,
        $sessions_frauds_columns VIEW raw,
        $sessions_staff_columns VIEW raw,
    )
);

$yandexuid_cluster = (
    SELECT
        yandexuid,
        cluster,
    FROM $prism_path
);

$icookie_cluster = (
    SELECT
        prism.cluster AS cluster,
        matching.target_id AS icookie,
    FROM $yandexuid_cluster AS prism
    LEFT JOIN $yandexuid_icookie_matching AS matching
    ON prism.yandexuid == matching.id
);

$active_icookie_cluster = (
    SELECT
        keys.icookie ?? '' AS icookie,
        CAST(CAST(yandexuid_cluster.cluster ?? icookie_cluster.cluster AS Int32) - 1 AS String) AS cluster,
    FROM $active_icookie AS keys
    LEFT JOIN $yandexuid_cluster AS yandexuid_cluster
    ON keys.icookie == yandexuid_cluster.yandexuid
    LEFT JOIN ANY $icookie_cluster AS icookie_cluster
    ON keys.icookie == icookie_cluster.icookie
);

-- Features
$get_device_type = ($user_agent_raw) -> {
    $user_agent_data = UserAgent::Parse($user_agent_raw);
    RETURN IF (
        $user_agent_data.isTablet,
        'tablet',
        IF (
            $user_agent_data.isTouch OR $user_agent_data.isMobile,
            'phone',
            'desktop',
        )
    );
};

$icookie_features = (
    SELECT
        SUBSTRING(key, 1) AS icookie,
        SOME(UserAgentRaw) AS user_agent_raw,
        SOME(UserRegion) AS user_region,
        SOME(Ip) AS ip,
        SOME(Rearr.SzmScreen) AS szm_screen,
        SOME(Rearr.SzmRatio) AS szm_ratio,
    FROM CONCAT(
        $sessions_clean_columns,
        $sessions_staff_columns,
    )
    WITH COLUMNS Struct<key:String>
    GROUP BY key
);

$icookie_features = (
    SELECT
        icookie,
        UserAgent::Parse(user_agent_raw).BrowserName AS browser,
        UserAgent::Parse(user_agent_raw).DeviceVendor AS device_vendor,
        UserAgent::Parse(user_agent_raw).OSFamily AS os_family,
        UserAgent::Parse(user_agent_raw).DeviceName AS device_name,
        COALESCE(user_region, Geo::RegionByIp(ip).id) AS region,
        $get_device_type(user_agent_raw) AS device_type,
        $get_screen_info(szm_screen, szm_ratio) AS screen_info,
    FROM $icookie_features
    WHERE user_agent_raw IS NOT NULL
        AND (user_region IS NOT NULL OR ip IS NOT NULL)
);

-- Join features and clusters
$features_and_cluster_by_user = (
    SELECT
        'y' || features.icookie AS icookie,
        cluster.cluster AS cluster,
        '{{ date }}' AS `date`,
        features.*
    WITHOUT
        features.icookie,
    FROM $icookie_features AS features
    LEFT JOIN ANY $active_icookie_cluster AS cluster
    USING(icookie)
);

INSERT INTO `{{ output_table }}` WITH TRUNCATE
SELECT *
FROM $features_and_cluster_by_user;
'''


def calculate(
    yt_client,
    yql_client,
    date,
    prism_weights=None,
    output_table=None,
):
    with NirvanaTransaction(yt_client) as transaction:
        prism_weights = prism_weights or os.path.join(config.PRISM_OFFLINE_USER_WEIGHTS_DIR, date)
        output_table = output_table or os.path.join(config.FEATURES_AND_CLUSTER_BY_USER_DIR, date)

        yql_client.execute(
            '\n'.join([
                metrics.prior_utils_query,
                templater.render_template(
                    features_and_cluster_by_user_query,
                    vars={
                        'user_sessions_clean': config.USER_SESSIONS_CLEAN_BY_DATE_TABLE.format(date),
                        'user_sessions_frauds': config.USER_SESSIONS_FRAUDS_BY_DATE_TABLE.format(date),
                        'user_sessions_staff': config.USER_SESSIONS_STAFF_BY_DATE_TABLE.format(date),
                        'prism_weights': prism_weights,
                        'yandexuid_icookie_matching': config.YANDEXUID_ICOOKIE_MATCHING_TABLE,
                        'output_table': output_table,
                        'date': date,
                    },
                ),
            ]),
            title='YQL Features and cluster by user calculation',
            transaction=str(transaction.transaction_id),
        )

        yt_helpers.set_ttl(output_table, timedelta(days=config.FEATURES_AND_CLUSTER_BY_USER_TTL_DAYS), yt_client=yt_client)
