import logging
import os

from crypta.lib.python import templater
from crypta.lib.python.nirvana.nirvana_helpers.nirvana_transaction import NirvanaTransaction
from crypta.lib.python.yt import yt_helpers
from crypta.prism.lib.config import config
from crypta.prism.lib.utils import metrics as metric_utils
from crypta.profile.lib import date_helpers


logger = logging.getLogger(__name__)

priors_key = 'browser, device_name, device_vendor, os_family, region, screen_info'

priors_preparation_query = '''
${{ id }}_prepared = (
    SELECT
        {{ key }},
        `count` AS `value`,
    FROM `{{ input_table }}`
);
'''

correlation_with_offline_query = '''
$features_and_cluster_by_user = (
    SELECT
        icookie,
        browser,
        device_vendor,
        os_family,
        device_name,
        region,
        screen_info,
        cluster AS offline_cluster,
    FROM `{{ features_and_cluster_by_user }}`
    WHERE cluster IS NOT NULL
);

$features_and_cluster_by_user_extended = (
    SELECT *
    FROM (
        SELECT
            icookie,
            offline_cluster,
            Geo::GetParents(CAST(region AS Int32)) AS region,
            $extend(browser) AS browser,
            $extend(device_vendor) AS device_vendor,
            $extend(os_family) AS os_family,
            $extend(device_name) AS device_name,
            $extend(screen_info) AS screen_info,
        FROM $features_and_cluster_by_user
    )
    FLATTEN LIST BY (
        browser,
        device_vendor,
        os_family,
        device_name,
        region,
        screen_info,
    )
);

$icookie_to_cluster = (
    SELECT
        icookie,
        SOME(offline_cluster) AS offline_cluster,
        MIN_BY(prior_cluster, `count`) AS prior_cluster,
    FROM (
        SELECT
            features_and_cluster_by_user.icookie AS icookie,
            features_and_cluster_by_user.offline_cluster AS offline_cluster,
            CAST(priors.prior_cluster AS String) AS prior_cluster,
            priors.`count` AS `count`,
        FROM $features_and_cluster_by_user_extended AS features_and_cluster_by_user
        LEFT JOIN ANY `{{ priors }}` AS priors
        USING(browser, device_vendor, os_family, device_name, region, screen_info)
    )
    GROUP BY icookie
);

INSERT INTO `{{ output_table }}` WITH TRUNCATE
SELECT
    'correlation_with_offline' AS source,
    COUNT(*) AS matched_count,
    CORRELATION(CAST(offline_cluster AS Uint64), CAST(prior_cluster AS Uint64)) AS `correlation`,
FROM $icookie_to_cluster;
'''


def get_day_to_day_correlation_query(
    date,
    day_to_day_correlation_output,
):
    return '\n'.join([
        templater.render_template(
            priors_preparation_query,
            vars={
                'id': 'first',
                'input_table': os.path.join(config.PRISM_PRIORS_DIR, date),
                'key': priors_key,
            },
        ),
        templater.render_template(
            priors_preparation_query,
            vars={
                'id': 'second',
                'input_table': os.path.join(
                    config.PRISM_PRIORS_DIR,
                    date_helpers.get_date_from_past(current_date=date, days=1),
                ),
                'key': priors_key,
            },
        ),
        templater.render_template(
            metric_utils.correlation_query,
            vars={
                'output_table': day_to_day_correlation_output,
                'key': priors_key,
            },
        ),
    ])


def get_correlation_with_offline_query(date, correlation_with_offline_output):
    return '\n'.join([
        metric_utils.prior_utils_query,
        templater.render_template(
            correlation_with_offline_query,
            vars={
                'features_and_cluster_by_user': os.path.join(config.FEATURES_AND_CLUSTER_BY_USER_DIR, date),
                'priors': os.path.join(config.PRISM_PRIORS_DIR, date),
                'output_table': correlation_with_offline_output,
            },
        ),
    ])


def send_correlation_metrics(yt_client, yql_client, date):
    with NirvanaTransaction(yt_client) as transaction, \
            yt_client.TempTable() as day_to_day_correlation_output, \
            yt_client.TempTable() as correlation_with_offline_output:
        yql_client.execute(
            '\n'.join([
                get_day_to_day_correlation_query(date, day_to_day_correlation_output),
                get_correlation_with_offline_query(date, correlation_with_offline_output),
            ]),
            title='YQL Calculate correlation',
            transaction=str(transaction.transaction_id),
        )

        metrics = [
            {
                'source': source,
                'matched_count': metrics_for_source['matched_count'],
                'correlation': metrics_for_source['correlation'],
            } for source, metrics_for_source in zip(
                [
                    'day_to_day_correlation',
                    'correlation_with_offline',
                ],
                [
                    next(yt_client.read_table(day_to_day_correlation_output)),
                    next(yt_client.read_table(correlation_with_offline_output)),
                ],
            )
        ]

        yt_helpers.write_stats_to_yt(
            yt_client=yt_client,
            table_path=config.PRIORS_QUALITY_METRICS,
            data_to_write=metrics,
            schema={
                'source': 'string',
                'matched_count': 'uint64',
                'correlation': 'double',
            },
            date=date,
            fielddate='fielddate',
        )

        metric_utils.report_correlation_metrics(
            yt_client=yt_client,
            yql_client=yql_client,
            transaction=transaction,
            metrics=metrics,
            metrics_table=config.PRIORS_QUALITY_METRICS,
            service_name='priors_correlation',
        )
