import datetime
import functools
import logging

from crypta.lib.python.logging import logging_helpers
from crypta.prism.lib.nirvana import utils as nirvana_utils
from crypta.prism.services.training.lib import (
    catboost_train_sample,
    clusters,
    features_mapping,
    raw_train_sample,
    roc_auc,
    thresholds,
)

import nirvana.job_context as nv


logger = logging.getLogger(__name__)


def main():
    logging_helpers.configure_stderr_logger(logging.getLogger(), level=logging.INFO)
    logger.info('Realtime prism training')

    context = nv.context()
    parameters = context.parameters
    inputs = context.get_inputs()
    outputs = context.get_outputs()
    json_input = inputs.get('json_input') if inputs.has('json_input') else None
    result_output = outputs.get('result')

    yt_client = nirvana_utils.get_yt_client(nv_params=parameters)
    yql_client = nirvana_utils.get_yql_client(nv_params=parameters)

    date = str(datetime.date.today())

    tasks_dict = {
        'prepare_catboost_train_sample': functools.partial(
            catboost_train_sample.prepare,
            yt_client=yt_client,
        ),
        'match_clusters': functools.partial(
            clusters.match,
            yt_client=yt_client,
            output=result_output,
        ),
        'compute_features_mapping': functools.partial(
            features_mapping.compute,
            yt_client=yt_client,
            yql_client=yql_client,
            output=result_output,
        ),
        'get_raw_train_sample': functools.partial(
            raw_train_sample.get,
            yt_client=yt_client,
            yql_client=yql_client,
        ),
        'get_roc_auc': functools.partial(
            roc_auc.get,
            yt_client=yt_client,
            json_input=json_input,
            date=date,
        ),
        'find_thresholds': functools.partial(
            thresholds.find,
            yt_client=yt_client,
            yql_client=yql_client,
            date=date,
            output=result_output,
        ),
    }

    job_name = parameters.get('job_name')
    logger.info('Job name: {}'.format(job_name))

    if job_name in tasks_dict:
        tasks_dict[job_name]()
    else:
        logger.warn('Unknown job_name "{}"'.format(job_name))
        exit(1)


if __name__ == '__main__':
    main()
