#!/usr/bin/env python
# -*- coding: utf-8 -*-
import collections

import tvmauth
from yabs.proto import user_profile_pb2

from crypta.lib.python.retryable_http_client import RetryableHttpClient
from crypta.lib.python.tvm.helpers import get_tvm_headers
from crypta.profile.lib import profile_helpers


# https://abc.yandex-team.ru/services/bigb/resources/?show-resource=3392495
BIGB_TVM_ID = 2001337
BIGB_URL = 'http://bigb-fast.yandex.ru/'


bb_keyword_id_to_keyword_name = {
    175: 'age_segments',
    198: 'top_common_site_ids',
    216: 'heuristic_segments',
    217: 'probabilistic_segments',
    220: 'yandex_loyalty',
    281: 'marketing_segments',
    544: 'lal_common',
    545: 'lal_private',
    546: 'lal_internal',
    547: 'heuristic_common',
    548: 'heuristic_private',
    549: 'heuristic_internal',
    569: 'exact_socdem',
    595: 'affinitive_site_ids',
    601: 'longterm_interests',

    877: 'offline_gender',
    878: 'offline_user_age_6s',
    879: 'offline_income_segments',
    880: 'offline_income_5_segments',

    885: 'offline_exact_socdem',
    886: 'offline_exact_socdem',
    887: 'offline_exact_socdem',
    888: 'offline_exact_socdem',

    174: 'gender',
    176: 'income_segments',
    543: 'user_age_6s',
    614: 'income_5_segments',

    1084: 'trainable_segments',
}
keyword_name_to_bb_keyword_id = {bb_keyword_id_to_keyword_name[k]: k for k in bb_keyword_id_to_keyword_name}

named_values = {
    174: profile_helpers.gender_id_to_name,
    175: profile_helpers.five_segment_age_id_to_name,
    543: profile_helpers.six_segment_age_id_to_name,
    176: profile_helpers.three_segment_income_id_to_name,
    614: profile_helpers.five_segment_income_id_to_name,

    877: profile_helpers.gender_id_to_name,
    878: profile_helpers.six_segment_age_id_to_name,
    879: profile_helpers.three_segment_income_id_to_name,
    880: profile_helpers.five_segment_income_id_to_name,

    885: profile_helpers.gender_id_to_name,
    886: profile_helpers.six_segment_age_id_to_name,
    887: profile_helpers.three_segment_income_id_to_name,
    888: profile_helpers.five_segment_income_id_to_name,
}

exact_socdem_keyword_id_to_field_name = {
    885: 'gender',
    886: 'age_segment',
    887: 'income_segment',
    888: 'income_5_segment',

    174: 'gender',
    543: 'age_segment',
    176: 'income_segment',
    614: 'income_5_segment',
}


def get_keyword_name(keyword_id, segment_id=None):
    if keyword_id == 217 and (segment_id is not None) and (segment_id in profile_helpers.interests_composite_segment_ids):
        return 'interests_composite'

    return bb_keyword_id_to_keyword_name[keyword_id]


def convert_exact_socdem_from_bb_to_profiles(pairs):
    exact_socdem_dict = {}

    for pair in pairs:
        key, value = pair.first, pair.second
        exact_socdem_dict[exact_socdem_keyword_id_to_field_name[key]] = named_values[key][value]

    return exact_socdem_dict


def parse_bb_record(bb_record):
    keyword_update_times = collections.defaultdict(collections.Counter)
    profile = collections.defaultdict(dict)

    for item in bb_record.items:
        keyword_id = item.keyword_id
        if keyword_id not in bb_keyword_id_to_keyword_name:
            continue

        keyword_name = get_keyword_name(keyword_id)
        keyword_update_times[keyword_id][item.update_time] += 1

        if keyword_id in (198, 547, 548, 549, 601):
            profile[keyword_name] = [int(x) for x in item.uint_values]
        elif keyword_id in (885, 886, 887, 888):
            profile[keyword_name][exact_socdem_keyword_id_to_field_name[keyword_id]] = named_values[keyword_id][item.uint_values[0]]
        elif keyword_id == 220:
            profile[keyword_name] = float(item.uint_values[0]) / 1000000
        elif keyword_id == 569:
            profile[keyword_name] = convert_exact_socdem_from_bb_to_profiles(item.pair_values)
        elif keyword_id in (216, 595):
            for pair in item.pair_values:
                segment_id, segment_value = str(pair.first), pair.second
                if keyword_id == 216:
                    profile[keyword_name][segment_id] = segment_value
                elif keyword_id == 595:
                    profile[keyword_name][segment_id] = float(segment_value) / 1000000
        elif keyword_id in (877, 878, 879, 880, 174, 175, 176, 543, 614, 281, 544, 545, 546, 1084):
            for value in item.weighted_uint_values:
                keyword_value = str(value.first)
                if keyword_id in named_values:
                    keyword_value = named_values[keyword_id][value.first]
                profile[keyword_name][keyword_value] = float(value.weight) / 1000000
        elif keyword_id == 217:
            for value in item.weighted_pair_values:
                segment_id, segment_group = str(value.first), str(value.second)
                keyword_name = get_keyword_name(keyword_id, segment_id)
                if segment_id not in profile[keyword_name]:
                    profile[keyword_name][segment_id] = {}
                profile[keyword_name][segment_id][segment_group] = float(value.weight) / 1000000

    return dict(profile), dict(keyword_update_times)


class BigbClient(RetryableHttpClient):
    def __init__(self, tmv_id, tvm_secret, logger, timeout=3, n_retries=3):
        super(BigbClient, self).__init__(n_retries, timeout)
        self._tvm_client = None

        self.tmv_id = tmv_id
        self.tvm_secret = tvm_secret
        self.logger = logger

        self.id_type_bb_name_dict = {
            'yandexuid': 'bigb-uid',
            'crypta_id': 'bigb-uid',
            'gaid': 'gaid',
            'idfa': 'idfa',
            'uuid': 'uuid',
            'puid': 'puid',
        }

    @property
    def tvm_client(self):
        # workaround for an error "Creating TvmClient is forbidden before fork."
        if self._tvm_client is None:
            self._tvm_client = tvmauth.TvmClient(tvmauth.TvmApiClientSettings(
                self_tvm_id=self.tmv_id,
                self_secret=self.tvm_secret,
                dsts=[BIGB_TVM_ID],
            ))
        return self._tvm_client

    def get_id_info(self, id, id_type, glue=False):
        assert id_type in self.id_type_bb_name_dict

        try:
            response = self._make_get_request(
                BIGB_URL,
                params={
                    self.id_type_bb_name_dict[id_type]: id,
                    'client': 'crypta-test',
                    'glue': int(glue),
                },
                headers=get_tvm_headers(self.tvm_client.get_service_ticket_for(tvm_id=BIGB_TVM_ID)),
            )

            decoded_response = user_profile_pb2.Profile()
            decoded_response.ParseFromString(response.content)
            return decoded_response
        except:
            self.logger.exception("failed to get profile for %s: %s", id_type, id)
            return user_profile_pb2.Profile()

    def get_parsed_bb_profile(self, id, id_type, glue=False):
        parsed_response = self.get_id_info(id, id_type, glue)

        if parsed_response and parsed_response.is_full:
            return parse_bb_record(parsed_response)
        else:
            return {}, {}
