#!/usr/bin/env python
# -*- coding: utf-8 -*-
import calendar
import datetime

from dateutil import relativedelta
import pytz
import six

DATE_FORMAT = '%Y-%m-%d'
DATETIME_FORMAT = '%Y-%m-%d %H:%M:%S'
YT_DATETIME_FORMAT = '%Y-%m-%dT%H:%M:%S.%fZ'
MSK = pytz.timezone("Europe/Moscow")


def get_today_date_string():
    return str(datetime.date.today())


def to_date_string(date, date_format=DATE_FORMAT):
    if isinstance(date, six.string_types):
        return date
    else:
        return date.strftime(date_format)


def from_date_string_to_datetime(date, date_format):
    return datetime.datetime.strptime(date, date_format)


def from_date_string_to_msk_datetime(date, date_format):
    return MSK.localize(from_date_string_to_datetime(date, date_format))


def from_datetime_to_timestamp(date):
    if date.tzinfo:
        epoch = datetime.datetime(1970, 1, 1, tzinfo=pytz.UTC)
        diff = date.astimezone(pytz.UTC) - epoch
    else:
        epoch = datetime.datetime(1970, 1, 1)
        diff = date - epoch
    return int(diff.total_seconds())


def from_date_string_to_timestamp(date_string):
    return from_datetime_to_timestamp(from_date_string_to_datetime(date_string, DATE_FORMAT))


def from_date_string_to_msk_timestamp(date_string):
    return from_datetime_to_timestamp(from_date_string_to_msk_datetime(date_string, DATE_FORMAT))


# TODO(ermolmak): remove it in favor of from_timestamp_to_moscow_datetime
def from_timestamp_to_datetime(timestamp):
    return datetime.datetime.fromtimestamp(int(timestamp))


def from_timestamp_to_moscow_datetime(timestamp):
    return datetime.datetime.fromtimestamp(int(timestamp), tz=MSK)


def get_date_from_past(current_date, days=0, months=0, years=0):
    if isinstance(current_date, six.string_types):
        return to_date_string(
            from_date_string_to_datetime(current_date, DATE_FORMAT)
            - relativedelta.relativedelta(days=days, months=months, years=years)
        )
    elif isinstance(current_date, datetime.date):
        return current_date - relativedelta.relativedelta(days=days, months=months, years=years)
    elif isinstance(current_date, datetime.datetime):
        return current_date - relativedelta.relativedelta(days=days, months=months, years=years)


def get_date_from_future(current_date, days=0, months=0, years=0):
    return get_date_from_past(current_date, -days, -months, -years)


def get_yesterday(current_date):
    return get_date_from_past(current_date, days=1)


def get_tomorrow(current_date):
    return get_date_from_future(current_date, days=1)


def from_utc_datetime_to_timestamp(date):
    return int(calendar.timegm(date.timetuple()))


def from_utc_date_string_to_timestamp(date_string):
    return from_utc_datetime_to_timestamp(from_date_string_to_datetime(date_string, DATE_FORMAT))


def from_utc_date_string_to_noon_timestamp(date_string):
    return from_utc_datetime_to_timestamp(
        from_date_string_to_datetime(date_string, DATE_FORMAT)
        + datetime.timedelta(hours=12)
    )


def generate_date_strings(start_date_string, end_date_string):
    start_date = from_date_string_to_datetime(start_date_string, DATE_FORMAT)
    end_date = from_date_string_to_datetime(end_date_string, DATE_FORMAT)

    date_strings = []
    while end_date - start_date >= datetime.timedelta(days=0):
        date_strings.append(to_date_string(start_date))
        start_date += datetime.timedelta(days=1)

    return date_strings


def generate_back_dates(end_date, back_days):
    return generate_date_strings(
        get_date_from_past(end_date, days=back_days - 1),
        end_date,
    )
