#!/usr/bin/env python
# -*- coding: utf-8 -*-
import collections
import functools
import logging
import os

import luigi

from crypta.profile.lib import date_helpers

logger = logging.getLogger(__name__)


def setup_external_tasks_retry():
    # retry tasks with missing external tables while some other tasks are running
    luigi.worker.worker.retry_external_tasks = luigi.BoolParameter(default=True)
    # retry failed tasks every 5 minutes (default is 15)
    luigi.scheduler.scheduler.retry_delay = luigi.IntParameter(default=5 * 60)

# TODO yt=None as an option


AttributeCondition = collections.namedtuple('AttributeCondition', 'attribute_name, func')


class YtNodeTarget(luigi.Target):
    def __init__(self, path, yt, attribute_conditions=None):
        self.path = path
        self.yt = yt
        self.attribute_conditions = attribute_conditions

    def exists(self):
        if not self.yt.exists(self.path):
            return False

        if self.attribute_conditions is None:
            return True

        for condition in self.attribute_conditions:
            attribute_value = self.yt.get_attribute(self.path, condition.attibute_name, None)
            if attribute_value is None or not condition.func(attribute_value):
                return False

        return True


def non_empty_table(row_count, allow_empty):
    return allow_empty or row_count


def attribute_equals(real_attribute_value, attribute_value):
    return real_attribute_value == attribute_value


def recently_updated(real_update_date, update_date):
    return update_date <= real_update_date[:10]


is_table_condition = AttributeCondition(
    attribute_name='type',
    func=functools.partial(attribute_equals, attribute_value='table'),
)


class YtNodeHasAttributeTarget(YtNodeTarget):
    def __init__(self, path, yt, attribute_name, attribute_value):
        super(YtNodeHasAttributeTarget, self).__init__(
            path,
            yt,
            attribute_conditions=[
                AttributeCondition(
                    attribute_name=attribute_name,
                    func=functools.partial(attribute_equals, attribute_value=attribute_value),
                ),
            ],
        )
        self.attribute_name = attribute_name
        self.attribute_value = attribute_value


class YtStaticTableTarget(YtNodeTarget):
    def __init__(self, path, yt, allow_empty=False):
        super(YtStaticTableTarget, self).__init__(
            path,
            yt,
            attribute_conditions=[
                is_table_condition,
                AttributeCondition(
                    attribute_name='row_count',
                    func=functools.partial(non_empty_table, allow_empty=allow_empty),
                ),
            ],
        )
        self.allow_empty = allow_empty


class ExternalYtStaticTableInput(luigi.ExternalTask):
    path = luigi.Parameter()
    yt = luigi.Parameter()
    allow_empty = luigi.BoolParameter(default=False)

    def output(self):
        return YtStaticTableTarget(self.path, self.yt, self.allow_empty)


class YtStaticTableHasAttributeTarget(YtNodeTarget):
    def __init__(self, path, yt, attribute_name, attribute_value, allow_empty=False):
        super(YtStaticTableHasAttributeTarget, self).__init__(
            path,
            yt,
            attribute_conditions=[
                is_table_condition,
                AttributeCondition(
                    attribute_name='row_count',
                    func=functools.partial(non_empty_table, allow_empty=allow_empty),
                ),
                AttributeCondition(
                    attribute_name=attribute_name,
                    func=functools.partial(attribute_equals, attribute_value=attribute_value),
                ),
            ],
        )
        self.attribute_name = attribute_name
        self.attribute_value = attribute_value
        self.allow_empty = allow_empty


class ExternalYtStaticTableHasAttributeInput(luigi.ExternalTask):
    path = luigi.Parameter()
    yt = luigi.Parameter()
    attribute_name = luigi.Parameter()
    attribute_value = luigi.Parameter()
    allow_empty = luigi.BoolParameter(default=False)

    def output(self):
        return YtStaticTableHasAttributeTarget(
            self.path,
            self.yt,
            self.attribute_name,
            self.attribute_value,
            self.allow_empty,
        )


class YtRewritableStaticTableTarget(YtNodeTarget):
    def __init__(self, path, yt, date, allow_empty=False, date_attribute='generate_date'):
        super(YtRewritableStaticTableTarget, self).__init__(
            path,
            yt,
            attribute_conditions=[
                is_table_condition,
                AttributeCondition(
                    attribute_name='row_count',
                    func=functools.partial(non_empty_table, allow_empty=allow_empty),
                ),
                AttributeCondition(
                    attribute_name=date_attribute,
                    func=functools.partial(recently_updated, update_date=date),
                ),
            ],
        )
        self.allow_empty = allow_empty
        self.date = date
        self.date_attribute = date_attribute


class ExternalYtRewritableStaticTableInput(luigi.ExternalTask):
    path = luigi.Parameter()
    yt = luigi.Parameter()
    date = luigi.Parameter()
    allow_empty = luigi.BoolParameter(default=False)
    date_attribute = luigi.Parameter(default='generate_date')

    def output(self):
        return YtRewritableStaticTableTarget(
            self.path,
            self.yt,
            self.attribute_name,
            self.attribute_value,
            self.allow_empty,
        )


# compatibility with old classes
ExternalInput = ExternalYtStaticTableInput
ExternalInputDate = ExternalYtRewritableStaticTableInput
AttributeExternalInput = ExternalYtStaticTableHasAttributeInput


# cleaners

class NoOldNodesByNameTarget(luigi.Target):
    def __init__(self, date, directory, ttl, yt, date_format=date_helpers.DATE_FORMAT):
        self.directory = directory
        self.date_format = date_format
        self.date = date
        self.ttl = ttl
        self.yt = yt

        self.last_datetime = date_helpers.get_date_from_past(
            date_helpers.from_date_string_to_datetime(date, date_format=self.date_format),
            self.ttl,
        )

    def exists(self):
        return len(self.get_old_nodes()) == 0

    def get_old_nodes(self):
        return filter(self.is_old_node, self.yt.list(self.directory, absolute=True))

    def is_old_node(self, path):
        node_name = os.path.basename(path)
        try:
            node_datetime = date_helpers.from_date_string_to_datetime(node_name, self.date_format)
        except ValueError:
            return False

        return node_datetime < self.last_datetime


class OldNodesByNameCleaner(luigi.Task):
    date = luigi.Parameter()
    directory = luigi.Parameter()
    ttl = luigi.IntParameter()
    yt = luigi.Parameter()
    date_format = luigi.Parameter(default=date_helpers.DATE_FORMAT)

    def run(self):
        for node_path in self.output().get_old_nodes():
            try:
                self.yt.remove(node_path, recursive=True, force=True)
            except Exception as err:
                logger.info('Failed to remove node {}: {}'.format(node_path, err.message))
                pass

    def output(self):
        return NoOldNodesByNameTarget(
            date=self.date,
            directory=self.directory,
            ttl=self.ttl,
            yt=self.yt,
            date_format=self.date_format,
        )
