# -*- coding: utf-8 -*-

import collections
import os
import urllib
import urlparse


CounterAndGoalCondition = collections.namedtuple('CounterAndGoalCondition', ['counter_id', 'goal_id'])
UrlCondition = collections.namedtuple('UrlCondition', ['host', 'path'])


def normalize_int_list_with_comments(values):
    resulting_values = []

    for value in values:
        normalized_value = value.split('#', 1)[0]  # strip comment
        normalized_value = normalized_value.strip()
        if normalized_value.isdigit():
            resulting_values.append(int(normalized_value))

    return resulting_values


def normalize_categories(values):
    return normalize_int_list_with_comments(values)


def normalize_artists(values):
    return normalize_int_list_with_comments(values)


def normalize_bundle_id(bundle_id):
    if not isinstance(bundle_id, basestring):
        return None
    return bundle_id.lower()


def normalize_counters_and_goals(values):
    normalized_conditions = []

    for value in values:
        normalized_value = value.split('#', 1)[0]  # strip comment
        normalized_value = normalized_value.strip()

        counter_id_and_goal_id = normalized_value.split(':', 1)
        if len(counter_id_and_goal_id) == 2:
            counter_id, goal_id = counter_id_and_goal_id
        elif len(counter_id_and_goal_id) == 1:
            counter_id = counter_id_and_goal_id[0]
            goal_id = None
        else:
            continue

        if not counter_id.isdigit():
            continue

        counter_id = int(counter_id)

        if goal_id is not None:
            if goal_id.isdigit():
                goal_id = int(goal_id)
            else:
                continue

        normalized_conditions.append(CounterAndGoalCondition(counter_id, goal_id))

    return normalized_conditions


def normalize_host(host):
    if not isinstance(host, basestring):
        return None
    parsed = urlparse.urlparse('//' + host)
    host = parsed.netloc.encode('idna')
    if not host:
        return None
    return host


def split_url(url):
    if not isinstance(url, basestring):
        return None

    if isinstance(url, str):
        url = url.decode('utf-8')

    if url.startswith('regexp:'):
        parts = url.split(':', 1)[1].split('/', 1)
        if len(parts) < 2:
            return None

        host, path_regexp = parts
        path_regexp = path_regexp.replace('\\', '\\\\')

        return UrlCondition(host, os.path.join(host.replace(u".", ur"\\."), path_regexp))

    parsed = urlparse.urlparse(u'//' + url)
    path = urllib.quote(parsed.path.strip(u'/').encode('utf-8'))
    host = parsed.netloc.encode('idna')

    if not host:
        return None

    return UrlCondition(host, os.path.join(host, path).replace(".", r"\\.") + ".*")


def normalize_values_with_func(values, func):
    normalized_values = []

    for value in values:
        normalized = func(value)
        if normalized is not None:
            normalized_values.append(normalized)

    return normalized_values


def normalize_apps(values):
    return normalize_values_with_func(values, normalize_bundle_id)


def split_urls(values):
    return normalize_values_with_func(values, split_url)


def normalize_hosts(values):
    return normalize_values_with_func(values, normalize_host)
