#!/usr/bin/env python
# -*- coding: utf-8 -*-

"""
Crypta user profile builder

Usage:
  main.py [--log-file-path=<path>] [--date=<date>]
  main.py precalculate_affinity [--log-file-path=<path>] [--date=<date>]
  main.py clean_up_tmp [--log-file-path=<path>] [--date=<date>]
  main.py upload_segments_to_audience [--log-file-path=<path>]
  main.py (-h | --help)

Options:
  main                          Main export profile task
  precalculate_affinity         Run precalculate affinity separately
  clean_up_tmp                  Clean tmp directory
  upload_segments_to_audience   Upload segments to audience
  --log-file-path=<path>        Path for logs
  --date=<date>                 Run date
"""

import datetime

from docopt import docopt
import luigi
from luigi import execution_summary

from crypta.profile.lib import (
    date_helpers,
    luigi_helpers,
)
from crypta.profile.runners.export_profiles.lib.affinity.precalculate_affinity import PrecalculateAffinity
from crypta.profile.runners.export_profiles.lib.export.check_audience_upload import CheckAudienceUpload
from crypta.profile.runners.export_profiles.lib.export.get_export_devid_profiles import GetExportDevidProfiles
from crypta.profile.runners.export_profiles.lib.export.get_visited_sites_for_dit import GetVisitedSitesForDit
from crypta.profile.runners.export_profiles.lib.export.upload_lab_segments_info_to_yt import UploadLabSegmentsInfoToYt
from crypta.profile.runners.export_profiles.lib.export.upload_profiles_to_logbroker import UploadProfilesToLogbroker
from crypta.profile.tasks.features.calculate_id_vectors import GetCryptaIdVectors
from crypta.profile.tasks.features.get_app_metrica_data import GetAppIdf, GetMonthlyDevidByApp, UpdateMonthlyDevidStorage
from crypta.profile.runners.export_profiles.lib.profiles_generation.get_affinitive_and_top_sites import (
    GetAffinitiveAndTopSitesYandexuid,
    GetAffinitiveAndTopSitesCryptaid,
)
from crypta.profile.runners.export_profiles.lib.profiles_generation.get_crypta_id_profiles_log import ReportCryptaIdProfilesLogCreation
from crypta.profile.runners.export_profiles.lib.profiles_generation.get_statbox_crypta_profiles_log import (
    GetStatboxCryptaProfilesLog,
    ReportYandexuidProfilesLogCreation,
)
from crypta.profile.utils import luigi_utils
from crypta.profile.utils.config import config
from crypta.profile.utils.loggers import get_file_logger
from crypta.profile.utils.yt_utils import get_yt_client


LUIGI_SUCCESS_STATUSES = (
    execution_summary.LuigiStatusCode.SUCCESS,
    execution_summary.LuigiStatusCode.SUCCESS_WITH_RETRY,
)


def clean_tmp_directory(date, logger=None, yt=None):
    if yt is None:
        yt = get_yt_client()

    if logger is None:
        logger, log_file_path = get_file_logger(
            name='clean_up',
            directory=config.TASKS_LOGS_DIRECTORY,
        )

    # tmp
    current_date = date_helpers.from_date_string_to_datetime(date, date_helpers.DATE_FORMAT)
    last_date = current_date - datetime.timedelta(days=2)

    for node in yt.search(config.PROFILES_TMP_YT_DIRECTORY, node_type=['table', 'file']):
        try:
            modification_time = datetime.datetime.strptime(
                yt.get_attribute(node, 'modification_time'),
                '%Y-%m-%dT%H:%M:%S.%fZ',
            )
            if modification_time < last_date:
                yt.remove(node, recursive=True)
        except Exception as error:
            logger.info('Tried to delete {}'.format(node))
            logger.exception(error.message)


class MainTask(luigi_utils.BaseYtTask):
    date = luigi.Parameter()
    task_group = 'export_profiles'

    def requires(self):
        today = self.date
        tasks = [
            GetStatboxCryptaProfilesLog(today),
            ReportYandexuidProfilesLogCreation(today),
            ReportCryptaIdProfilesLogCreation(today),
            UploadProfilesToLogbroker(today, 'yandexuid'),
            UploadProfilesToLogbroker(today, 'crypta_id'),
            GetAffinitiveAndTopSitesYandexuid(today),
            GetAffinitiveAndTopSitesCryptaid(today),
            GetExportDevidProfiles(today),
            GetVisitedSitesForDit(today),
            GetAppIdf(today),
            GetMonthlyDevidByApp(today),
            UpdateMonthlyDevidStorage(today, 'vector'),
            GetCryptaIdVectors(today),
        ]

        self.logger.info('Checking environment == {}'.format(config.environment))
        if config.environment == 'production':
            tasks.append(UploadLabSegmentsInfoToYt(today))

        return tasks

    def run(self):
        self.logger.info('Cleaning tmp up')
        clean_tmp_directory(self.date, logger=self.logger, yt=self.yt)
        self.logger.info('Export profiles process has finished successfully')

    def complete(self):
        return False


def main():
    arguments = docopt(__doc__)
    today = str(datetime.date.today())
    config.DEFAULT_POOL = config.EXPORT_PROFILES_POOL

    luigi_utils.configure_luigi_stdout_logger()
    luigi_helpers.setup_external_tasks_retry()

    if arguments['--log-file-path']:
        config.LOCAL_LOGS_DIRECTORY = config.TASKS_LOGS_DIRECTORY = arguments['--log-file-path']
        config.LOCAL_STORAGE_DIRECTORY = arguments['--log-file-path']

    if arguments['--date']:
        today = arguments['--date']

    def run(task_cls, workers, detailed_summary=False):
        return luigi.run(
            [
                '--scheduler-url', config.LUIGI_SCHEDULER_URL,
                '--workers', str(workers),
                '--date', today,
                '--log-level', 'INFO',
            ],
            main_task_cls=task_cls,
            detailed_summary=detailed_summary,
        )

    if arguments['precalculate_affinity']:
        run(PrecalculateAffinity, 3)
    elif arguments['clean_up_tmp']:
        clean_tmp_directory(today)
    elif arguments['upload_segments_to_audience']:
        run(CheckAudienceUpload, 2)
    else:
        luigi_run_result = run(MainTask, 7, detailed_summary=True)
        if luigi_run_result.status not in LUIGI_SUCCESS_STATUSES:
            exit(1)


if __name__ == '__main__':
    main()
