#!/usr/bin/env python
# -*- coding: utf-8 -*-
import collections

import luigi

from crypta.lib.python import templater
from crypta.profile.utils import luigi_utils
from crypta.profile.utils.config import config

MIN_ID_COUNT_WITH_SITE = 2

HostTables = collections.namedtuple('HostTables', ['idf', 'flattened_hits'])

YQL_QUERY_TEMPLATE = u"""
INSERT INTO `{{output_table}}` WITH TRUNCATE
SELECT *
FROM (
{% for source_type in ("metrica", "bar") %}
    SELECT
        flattened_hits.{{id_type}} AS {{id_type}},
        flattened_hits.host AS host,
        idf.weight AS weight
    FROM `{% if source_type == "bar" %}{{flattened_bar_hits}}{% else %}{{flattened_metrics_hits}}{% endif %}` AS flattened_hits
    INNER JOIN `{% if source_type == "bar" %}{{bar_idf}}{% else %}{{metrics_idf}}{% endif %}` AS idf
    ON flattened_hits.host == idf.host
    INNER JOIN `{{hosts_counter}}` AS hosts_counter
    ON idf.host == hosts_counter.site
    WHERE idf.count >= {{min_id_count_with_site}}
        AND {% if source_type == "bar"%}NOT {% endif %}(metrica_visitors_count ?? 0 >= bar_visitors_count ?? 0)
{{"\nUNION ALL\n\n" if not loop.last}}
{%- endfor -%}
)
ORDER BY {{id_type}};
"""


class PrecalculateHostsAffinity(luigi_utils.BaseYtTask):
    date = luigi.Parameter()
    id_type = luigi.Parameter()
    task_group = 'precalculate_affinity'

    def requires(self):
        bar_tables = HostTables(
            idf={
                config.CRYPTA_ID: config.CRYPTA_ID_BAR_IDF_TABLE,
                config.YANDEXUID: config.YANDEXUID_BAR_IDF_TABLE,
            },
            flattened_hits={
                config.CRYPTA_ID: config.CRYPTA_ID_BAR_FLATTENED_HITS_TABLE,
                config.YANDEXUID: config.YANDEXUID_BAR_FLATTENED_HITS_TABLE,
            },
        )

        metrics_tables = HostTables(
            idf={
                config.CRYPTA_ID: config.CRYPTA_ID_METRICS_IDF_TABLE,
                config.YANDEXUID: config.YANDEXUID_METRICS_IDF_TABLE,
            },
            flattened_hits={
                config.CRYPTA_ID: config.CRYPTA_ID_METRICS_FLATTENED_HITS_TABLE,
                config.YANDEXUID: config.YANDEXUID_METRICS_FLATTENED_HITS_TABLE,
            },
        )

        counter_tables = {
            config.CRYPTA_ID: config.CRYPTA_ID_METRICA_BROWSER_COUNTER_TABLE,
            config.YANDEXUID: config.YANDEXUID_METRICA_BROWSER_COUNTER_TABLE,
        }

        return {
            'metrics_idf': luigi_utils.ExternalInputDate(metrics_tables.idf[self.id_type], self.date),
            'bar_idf': luigi_utils.ExternalInputDate(bar_tables.idf[self.id_type], self.date),
            'flattened_metrics_hits': luigi_utils.ExternalInputDate(metrics_tables.flattened_hits[self.id_type], self.date),
            'flattened_bar_hits': luigi_utils.ExternalInputDate(bar_tables.flattened_hits[self.id_type], self.date),
            'hosts_counter': luigi_utils.ExternalInputDate(counter_tables[self.id_type], self.date),
        }

    def output(self):
        table = {
            config.CRYPTA_ID: config.CRYPTA_ID_HOSTS_AFFINITY,
            config.YANDEXUID: config.YANDEXUID_HOSTS_AFFINITY,
        }

        return luigi_utils.YtDailyRewritableTarget(table[self.id_type], self.date)

    def run(self):
        self.yt.config['spec_defaults']['pool'] = config.DEFAULT_POOL

        with self.yt.Transaction() as transaction:
            query = templater.render_template(
                YQL_QUERY_TEMPLATE,
                {
                    'flattened_metrics_hits': self.input()['flattened_metrics_hits'].table,
                    'flattened_bar_hits': self.input()['flattened_bar_hits'].table,
                    'metrics_idf': self.input()['metrics_idf'].table,
                    'bar_idf': self.input()['bar_idf'].table,
                    'hosts_counter': self.input()['hosts_counter'].table,
                    'output_table': self.output().table,
                    'id_type': self.id_type,
                    'min_id_count_with_site': MIN_ID_COUNT_WITH_SITE,
                }
            )
            self.yql.query(
                query_string=query,
                transaction=transaction,
            )

            self.yt.set_attribute(self.output().table, 'generate_date', self.date)
