#!/usr/bin/python -tt
# -*- coding: utf-8 -*-
import luigi
import numpy as np

from yt.wrapper import with_context

import crypta.lib.python.spine.consts.attributes as spine_attributes
import crypta.lib.python.yt.time_utils as yt_time_utils
from crypta.profile.lib import vector_helpers
from crypta.profile.utils.config import config
from crypta.profile.utils.luigi_utils import BaseYtTask, YtDailyRewritableTarget, YtTableAttributeTarget
from crypta.profile.utils.loggers import TimeTracker
from crypta.profile.runners.export_profiles.lib.export.get_daily_export_and_process_bb_storage import GetDailyExportAndProcessBbStorage
from crypta.profile.tasks.features.calculate_id_vectors import GetDailyYandexuidVectors


@with_context
def add_vector(key, records, context):
    profile_record = None
    vector_record = None
    for record in records:
        if not context.table_index:
            profile_record = record
        elif context.table_index == 1:
            vector_record = record

    if profile_record:
        if vector_record:
            profile_record['packed_vector'] = vector_record['packed_vector']
        yield profile_record


def pack_vectors(record):
    concatenated_normalized_vector = vector_helpers.vector_row_to_features(record)
    packed_vector = np.rint(concatenated_normalized_vector * 127).astype(np.int8)
    yield {'yandexuid': record['yandexuid'], 'packed_vector': packed_vector.tostring()}


class PackVectors(BaseYtTask):
    date = luigi.Parameter()
    priority = 90
    task_group = 'export_profiles'

    def requires(self):
        return GetDailyYandexuidVectors(self.date)

    def run(self):
        with self.yt.Transaction():
            self.yt.create_empty_table(
                self.output().table,
                schema={'yandexuid': 'uint64', 'packed_vector': 'string'},
            )

            self.yt.run_map(
                pack_vectors,
                self.input().table,
                self.output().table,
            )
            self.yt.run_sort(self.output().table, sort_by='yandexuid')
            self.yt.set_attribute(self.output().table, 'generate_date', self.date)

    def output(self):
        return YtDailyRewritableTarget(
            table=config.PACKED_VECTORS_TABLE,
            date=self.date
        )


class AddVectorsToDailyExport(BaseYtTask):
    date = luigi.Parameter()
    priority = 100
    task_group = 'export_profiles'

    def requires(self):
        return {
            'daily_export_and_bb_storage': GetDailyExportAndProcessBbStorage(self.date, 'yandexuid'),
            'packed_vectors': PackVectors(self.date),
        }

    def run(self):
        with TimeTracker(self.__class__.__name__):
            daily_export_table = self.input()['daily_export_and_bb_storage']['daily_export'].table
            with self.yt.Transaction(), self.yt.TempTable() as temporary_table:
                input_table_schema = self.yt.get_attribute(daily_export_table, 'schema')
                self.yt.create_empty_table(
                    temporary_table,
                    schema=input_table_schema + [{'name': 'packed_vector', 'type': 'string'}]
                )
                self.yt.run_reduce(
                    add_vector,
                    source_table=[
                        daily_export_table,
                        self.input()['packed_vectors'].table,
                    ],
                    destination_table=temporary_table,
                    reduce_by='yandexuid',
                )

                self.yt.remove(daily_export_table)
                self.yt.run_sort(
                    temporary_table,
                    daily_export_table,
                    sort_by='yandexuid',
                )

                output_table = self.output().table
                self.yt.set_attribute(output_table, 'vectors', self.date)
                self.yt.set_attribute(output_table, spine_attributes.GENERATE_DATETIME, yt_time_utils.get_current_yt_timestamp())

    def output(self):
        return YtTableAttributeTarget(
            table=self.input()['daily_export_and_bb_storage']['daily_export'].table,
            attribute_name='vectors',
            attribute_value=self.date,
        )
