#!/usr/bin/env python
# -*- coding: utf-8 -*-

import luigi
from yt.wrapper import with_context

from crypta.profile.utils.config import config
from crypta.profile.utils.loggers import TimeTracker
from crypta.profile.utils.luigi_utils import YtDailyRewritableTarget, BaseYtTask, ExternalInput

from crypta.profile.runners.export_profiles.lib.profiles_generation.get_socdem_by_crypta_id import GetSocdemByCryptaId


@with_context
def join_with_appmetrica_id(key, records, context):
    profiles_record = None
    appmetrica_devids = []
    for record in records:
        if context.table_index == 0:
            profiles_record = record
        elif context.table_index == 1:
            appmetrica_devids.append(record['mm_device_id'])

    if profiles_record and appmetrica_devids:
        for appmetrica_devid in appmetrica_devids:
            new_profiles_record = dict(profiles_record)
            new_profiles_record['appmetrica_devid'] = appmetrica_devid
            new_profiles_record['id'] = new_profiles_record['devid']
            yield new_profiles_record


class GetExportDevidProfiles(BaseYtTask):
    date = luigi.Parameter()
    task_group = 'export_profiles'

    def requires(self):
        return {
            'devid_hash': ExternalInput(config.DEV_HASH_MONTH),
            'corrected_profiles': GetSocdemByCryptaId(self.date),
        }

    def run(self):
        with TimeTracker(self.__class__.__name__):
            with self.yt.Transaction():
                self.yt.create_empty_table(
                    self.output().table,
                    schema={
                        'devid': 'string',
                        'id': 'string',
                        'id_type': 'string',
                        'appmetrica_devid': 'string',
                        'gender': 'any',
                        'user_age_6s': 'any',
                        'income_segments': 'any',
                        'income_5_segments': 'any',
                        'exact_socdem': 'any',
                    },
                )

                self.yt.run_reduce(
                    join_with_appmetrica_id,
                    source_table=(
                        self.yt.TablePath(
                            self.input()['corrected_profiles']['corrected_devid_profiles'].table,
                            columns=(
                                'devid', 'id_type', 'gender', 'user_age_6s', 'income_segments',
                                'income_5_segments', 'exact_socdem',
                            ),
                            rename_columns={'id': 'devid'},
                        ),
                        self.yt.TablePath(
                            self.input()['devid_hash'].table,
                            columns=('devid', 'mm_device_id'),
                            rename_columns={'id': 'devid'},
                        ),
                    ),
                    destination_table=self.output().table,
                    reduce_by='devid',
                )
                self.yt.run_sort(self.output().table, sort_by='devid')
                self.yt.set_attribute(self.output().table, 'generate_date', self.date)

    def output(self):
        return YtDailyRewritableTarget(config.DEVID_PROFILES_EXPORT_TABLE, date=self.date)
