import cProfile
import logging

import yt.wrapper as yt

from crypta.lib.python.logging import logging_helpers
from crypta.profile.lib import date_helpers
from crypta.profile.runners.export_profiles.lib.profiles_generation import postprocess_profiles
from crypta.profile.utils.config import config
from crypta.profile.utils.segment_utils import boolparser
from crypta.profile.utils.yt_utils import get_yt_client

logger = logging.getLogger(__name__)


def get_export_expressions_from_yt(yt_client):
    return {
        export["id"]: postprocess_profiles.Export(
            field_name=postprocess_profiles._get_segment_field_name(export.get("keyword_id"), export.get("segment_id", 0)),
            segment_id=export.get("segment_id", 0),
            expressions=u' OR '.join([expression["normalized"] for expression in export["full_expressions"]]) if export.get("full_expressions") else export["id"],
        )
        for segment in yt_client.read_table("//home/crypta/production/lab/database/Segment")
        for export in segment["exports"]["exports"]
        if export.get("state") in ('CREATED', 'ACTIVE', 'DEPRECATED', 'DISABLE_REQUESTED')
    }


def main():
    logging_helpers.configure_stdout_logger(logger)
    yt_client = get_yt_client()

    logger.info("Getting exports")
    exports_expressions = get_export_expressions_from_yt(yt_client)

    logger.info("Parsing exports")
    parser = boolparser.ExpressionParser(exports_expressions, logger)
    trees = parser.build_trees()

    date = "2021-09-03"

    current_time = date_helpers.from_utc_date_string_to_noon_timestamp(date)
    outdated_shortterm_interests_threshold = current_time - 24 * 3600 * config.SHORTTERM_INTERESTS_EXPIRE_DAYS

    rules_processor = postprocess_profiles.PostprocessProfilesMapper(
        trees=trees,
        exports_expressions=exports_expressions,
        outdated_shortterm_interests_threshold=outdated_shortterm_interests_threshold,
    )

    logger.info("Negatives: %s", rules_processor.negative_segments)

    logger.info("Reading input")
    path = yt.TablePath("//home/crypta/team/cherenkov-p-a/CRYPTA-15163-2021-09-03-joined-yandexuid", start_index=0, end_index=1000)
    rows = list(yt_client.read_table(path))

    logger.info("Begin profilings")
    pr = cProfile.Profile()
    pr.enable()

    for row in rows:
        list(rules_processor(row))

    pr.disable()

    logger.info("End profiling")

    pr.print_stats(sort='tottime')
