#!/usr/bin/env python
# -*- coding: utf-8 -*-
import os

import luigi
from yt.wrapper import with_context

from crypta.profile.utils.config import config
from crypta.profile.utils.loggers import TimeTracker
from crypta.profile.utils.utils import additional_yandexuid_attributes_schema
from crypta.profile.utils.luigi_utils import BaseYtTask, ExternalInput, YtTarget, OldNodesByNameCleaner

from crypta.profile.tasks.features.get_crypta_ids import GetCryptaIds
from crypta.profile.runners.export_profiles.lib.profiles_generation.calculate_loyalty import CalculateLoyalty
from crypta.profile.runners.export_profiles.lib.profiles_generation.get_basic_profiles import GetBasicProfiles
from crypta.profile.runners.export_profiles.lib.profiles_generation.get_affinitive_and_top_sites import GetAffinitiveAndTopSitesYandexuid


@with_context
def join_yandexuid_profiles(key, rows, context):
    is_active_yandexuid = False
    output_record = {
        'yandexuid': key['yandexuid'],
        'icookie': key['yandexuid'],
    }

    for row in rows:
        if not is_active_yandexuid and context.table_index == 0:
            is_active_yandexuid = True
            continue
        elif not is_active_yandexuid:
            return

        if 'icookie' in row:
            if row['icookie'] is not None:
                output_record['icookie'] = row['icookie']
        else:
            output_record.update(row)

    yield output_record


class GetAdditionalYandexuidAttributes(BaseYtTask):
    date = luigi.Parameter()
    priority = 100
    task_group = 'export_profiles'

    def requires(self):
        return {
            'socdem': GetBasicProfiles(self.date),
            'affinitive_and_top_sites': GetAffinitiveAndTopSitesYandexuid(self.date),
            'yandex_loyalty': CalculateLoyalty(self.date),
            'icookies': ExternalInput(config.YANDEXUID_TO_ICOOKIE),
            'crypta_ids': GetCryptaIds(self.date),
            'cleaner': OldNodesByNameCleaner(
                self.date,
                folder=config.ADDITIONAL_YANDEXUID_ATTRIBUTES_DIRECTORY,
                lifetime=config.NUMBER_OF_INTERMEDIATE_PROFILES_TABLES_TO_KEEP,
            ),
        }

    def output(self):
        return YtTarget(os.path.join(config.ADDITIONAL_YANDEXUID_ATTRIBUTES_DIRECTORY, self.date))

    def run(self):
        with TimeTracker(monitoring_name=self.__class__.__name__):
            with self.yt.Transaction():
                source_tables = [
                    self.yt.TablePath(self.input()['socdem'].table, columns=['yandexuid']),
                    self.yt.TablePath(self.input()['icookies'].table, columns=['yandexuid', 'icookie']),
                    self.input()['crypta_ids']['yandexuid_cryptaid'].table,
                    self.input()['yandex_loyalty'].table,
                    self.yt.TablePath(
                        self.input()['affinitive_and_top_sites']['daily_table'].table,
                        columns=['yandexuid', 'affinitive_site_ids', 'affinitive_sites',
                                 'top_common_sites', 'top_common_site_ids'],
                    ),
                ]

                for table in source_tables:
                    self.yt.sort_if_needed(
                        table,
                        sort_by='yandexuid',
                    )

                self.yt.create_empty_table(
                    self.output().table,
                    schema=additional_yandexuid_attributes_schema,
                )

                self.yt.run_reduce(
                    join_yandexuid_profiles,
                    source_tables,
                    self.output().table,
                    reduce_by='yandexuid',
                )

                self.yt.run_sort(
                    self.output().table,
                    sort_by='yandexuid',
                )
