#!/usr/bin/env python
# -*- coding: utf-8 -*-

import datetime
from os.path import join

import luigi

from crypta.lib.python.yt import yt_helpers
from crypta.profile.lib.socdem_helpers.inference_utils import inference
from crypta.profile.runners.export_profiles.lib.profiles_generation.get_merged_segments import CombineSegmentsByYandexuid
from crypta.profile.tasks.features.calculate_id_vectors import GetDailyYandexuidVectors
from crypta.profile.tasks.features.merge_hits_by_id import MergeHitsByYandexuid
from crypta.profile.utils.config import config
from crypta.profile.utils.loggers import (
    send_to_graphite,
    TimeTracker,
)
from crypta.profile.utils.luigi_utils import (
    BaseYtTask,
    YtTarget,
)


class GetBasicProfiles(BaseYtTask):
    date = luigi.Parameter()
    priority = 100
    task_group = 'export_profiles'
    resource_type = 'CRYPTA_SOCDEM_MODELS'

    def requires(self):
        return {
            'yandexuid_daily_vectors': GetDailyYandexuidVectors(date=self.date),
            'merged_segments': CombineSegmentsByYandexuid(date=self.date),
            'merged_hits_by_yandexuid': MergeHitsByYandexuid(date=self.date, data_source='metrics'),
        }

    def output(self):
        return YtTarget(join(config.RAW_YANDEXUID_PROFILES_YT_DIRECTORY, self.date))

    def run(self):
        with TimeTracker(self.__class__.__name__):
            with self.yt.TempTable() as neuro_raw_yandexuid_profiles:
                released = 'stable' if config.environment == 'production' else 'testing'

                inference.get_nn_predictions_for_all_profiles(
                    yt_client=self.yt,
                    is_mobile=False,
                    neuro_raw_profiles=neuro_raw_yandexuid_profiles,
                    logger=self.logger,
                    resource_type=self.resource_type,
                    released=released,
                    date=self.date,
                )
                inference.get_catboost_predictions_for_all_profiles(
                    yt_client=self.yt,
                    yql_client=self.yql.yql_client,
                    is_mobile=False,
                    neuro_raw_profiles=neuro_raw_yandexuid_profiles,
                    raw_profiles=self.output().table,
                    logger=self.logger,
                    resource_type=self.resource_type,
                    released=released,
                    date=self.date,
                )

                if config.environment == 'production':
                    send_to_graphite(
                        'classification.profiles',
                        self.yt.row_count(self.output().table),
                    )

                yt_helpers.set_ttl(
                    self.output().table,
                    datetime.timedelta(days=config.NUMBER_OF_INTERMEDIATE_PROFILES_TABLES_TO_KEEP),
                    yt_client=self.yt,
                )
