#!/usr/bin/env python
# -*- coding: utf-8 -*-
import json
import os

from library.python import resource
import luigi

from crypta.lib.python import templater
import crypta.lib.python.spine.consts.attributes as spine_attributes
import crypta.lib.python.yt.time_utils as yt_time_utils
from crypta.profile.lib import (
    date_helpers,
    profile_helpers,
)
from crypta.profile.runners.export_profiles.lib.profiles_generation.get_affinitive_and_top_sites import GetAffinitiveAndTopSitesCryptaid
from crypta.profile.runners.export_profiles.lib.profiles_generation.get_merged_segments import ExpandSegmentsStorage
from crypta.profile.runners.export_profiles.lib.profiles_generation.get_socdem_by_crypta_id import GetSocdemByCryptaId
from crypta.profile.runners.export_profiles.lib.profiles_generation.postprocess_profiles import (
    get_export_expressions,
    PostprocessProfilesMapper,
)
from crypta.profile.utils import (
    interests_helpers,
    reactor,
    utils,
)
from crypta.profile.utils.config import config
from crypta.profile.utils.loggers import TimeTracker
from crypta.profile.utils.luigi_utils import (
    BaseYtTask,
    ExternalInput,
    OldNodesByNameCleaner,
    YtTarget,
)
from crypta.profile.utils.segment_utils.boolparser import ExpressionParser


SECONDS_IN_A_DAY = 24 * 60 * 60


class GetCryptaIdProfilesLog(BaseYtTask):
    date = luigi.Parameter()
    priority = 100
    task_group = 'export_profiles'

    def requires(self):
        return {
            'segments': ExpandSegmentsStorage(self.date),
            'socdem': GetSocdemByCryptaId(self.date),
            'affinitive_and_top_sites': GetAffinitiveAndTopSitesCryptaid(self.date),
            'cryptaid_profiles_for_14days': ExternalInput(config.CRYPTAID_EXPORT_PROFILES_14_DAYS_TABLE),
            'cleaner': OldNodesByNameCleaner(
                date_helpers.get_yesterday(self.date),
                folder=config.CRYPTAID_PROFILES_EXPORT_DIRECTORY,
                lifetime=config.NUMBER_OF_INTERMEDIATE_PROFILES_TABLES_TO_KEEP,
            ),
        }

    def output(self):
        yesterday = date_helpers.get_yesterday(self.date)
        return YtTarget(os.path.join(config.CRYPTAID_PROFILES_EXPORT_DIRECTORY, yesterday))

    def run(self):
        output_table = self.output().table

        with TimeTracker(self.__class__.__name__):
            with self.yt.Transaction() as transaction, self.yt.TempTable() as joined_profiles:
                bb_keyword_id_to_fields = {str(key): value for key, value in
                                           utils.bb_keyword_id_to_field_name.iteritems()}
                del bb_keyword_id_to_fields['217']
                # indirect connection with BB keywords
                not_strict_bb_keyword_id_to_field_names = {
                    '198': 'top_common_sites',
                    '595': 'affinitive_sites',
                    '217': {
                        'probabilistic_segments': 'default',
                        'interests_composite': list(profile_helpers.interests_composite_segment_ids),
                    },
                }

                self.yt.create_empty_table(
                    output_table,
                    schema=utils.crypta_id_profiles_schema,
                    additional_attributes={
                        'bb_keywords_to_field_name': bb_keyword_id_to_fields,
                        'not_strict_bb_keyword_id_to_field_names': not_strict_bb_keyword_id_to_field_names,
                    },
                )

                query = templater.render_template(
                    resource.find('/query/join_cryptaid_profiles.yql'),
                    vars={
                        'update_time': str(date_helpers.from_utc_date_string_to_noon_timestamp(self.date)),
                        'socdem_cryptaid_table': self.input()['socdem']['crypta_id_profiles'].table,
                        'expanded_segments_by_crypta_id_table': self.input()['segments']['crypta_id'].table,
                        'affinitive_and_top_sites_cryptaid_table': self.input()['affinitive_and_top_sites']['daily_table'].table,
                        'cryptaid_profiles_for_14days_table': self.input()['cryptaid_profiles_for_14days'].table,
                        'output_table': joined_profiles,
                    },
                )
                self.yql.query(
                    query_string=query,
                    transaction=transaction,
                    title='YQL Join cryptaid profiles',
                )

                exports_expressions = get_export_expressions(interests_helpers.LabSegmentsInfo().segments)
                parser = ExpressionParser(exports_expressions, self.logger)
                trees = parser.build_trees()

                current_time = date_helpers.from_utc_date_string_to_noon_timestamp(self.date)
                outdated_shortterm_interests_threshold = current_time - SECONDS_IN_A_DAY * config.SHORTTERM_INTERESTS_EXPIRE_DAYS

                operation = self.yt.run_map(
                    PostprocessProfilesMapper(
                        trees=trees,
                        exports_expressions=exports_expressions,
                        outdated_shortterm_interests_threshold=outdated_shortterm_interests_threshold,
                    ),
                    source_table=joined_profiles,
                    destination_table=output_table,
                    spec={'data_size_per_job': config.POSTPROCESS_PROFILES_MAPPER_DATA_SIZE_PER_JOB},
                )
                self.logger.info(json.dumps(operation.get_job_statistics()))
                self.yt.sort_if_needed(output_table, sort_by='crypta_id')
                self.yt.set_attribute(output_table, spine_attributes.GENERATE_DATETIME, yt_time_utils.get_current_yt_timestamp())


class ReportCryptaIdProfilesLogCreation(reactor.ReportProfilesLogCreation):
    date = luigi.Parameter()

    def __init__(self, *args, **kwargs):
        super(ReportCryptaIdProfilesLogCreation, self).__init__(*args, **kwargs)
        self.namespace_path = '/crypta/profile/hahn/crypta-cryptaid-profiles-log'

    def requires(self):
        return GetCryptaIdProfilesLog(self.date)
